// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
import { getDefaultProxySettings } from "@azure/core-rest-pipeline";
import { isNode } from "@azure/core-util";
import { BlobServiceClient } from "@azure/storage-blob";
import { isPipelineLike, newPipeline } from "./Pipeline";
import { AnonymousCredential } from "@azure/storage-blob";
import { StorageSharedKeyCredential } from "./credentials/StorageSharedKeyCredential";
import { DataLakeFileSystemClient } from "./DataLakeFileSystemClient";
import { StorageClient } from "./StorageClient";
import { appendToURLPath, appendToURLQuery, extractConnectionStringParts, } from "./utils/utils.common";
import { toDfsEndpointUrl, toFileSystemPagedAsyncIterableIterator } from "./transforms";
import { tracingClient } from "./utils/tracing";
import { AccountSASPermissions } from "./sas/AccountSASPermissions";
import { generateAccountSASQueryParameters, generateAccountSASQueryParametersInternal, } from "./sas/AccountSASSignatureValues";
import { AccountSASServices } from "./sas/AccountSASServices";
/**
 * DataLakeServiceClient allows you to manipulate Azure
 * Data Lake service resources and file systems. The storage account provides
 * the top-level namespace for the Data Lake service.
 */
export class DataLakeServiceClient extends StorageClient {
    /**
     *
     * Creates an instance of DataLakeServiceClient from connection string.
     *
     * @param connectionString - Account connection string or a SAS connection string of an Azure storage account.
     *                                  [ Note - Account connection string can only be used in NODE.JS runtime. ]
     *                                  Account connection string example -
     *                                  `DefaultEndpointsProtocol=https;AccountName=myaccount;AccountKey=accountKey;EndpointSuffix=core.windows.net`
     *                                  SAS connection string example -
     *                                  `BlobEndpoint=https://myaccount.blob.core.windows.net/;QueueEndpoint=https://myaccount.queue.core.windows.net/;FileEndpoint=https://myaccount.file.core.windows.net/;TableEndpoint=https://myaccount.table.core.windows.net/;SharedAccessSignature=sasString`
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    static fromConnectionString(connectionString, 
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-options */
    options) {
        options = options || {};
        const extractedCreds = extractConnectionStringParts(connectionString);
        if (extractedCreds.kind === "AccountConnString") {
            if (isNode) {
                const sharedKeyCredential = new StorageSharedKeyCredential(extractedCreds.accountName, extractedCreds.accountKey);
                if (!options.proxyOptions) {
                    options.proxyOptions = getDefaultProxySettings(extractedCreds.proxyUri);
                }
                const pipeline = newPipeline(sharedKeyCredential, options);
                return new DataLakeServiceClient(toDfsEndpointUrl(extractedCreds.url), pipeline);
            }
            else {
                throw new Error("Account connection string is only supported in Node.js environment");
            }
        }
        else if (extractedCreds.kind === "SASConnString") {
            const pipeline = newPipeline(new AnonymousCredential(), options);
            return new DataLakeServiceClient(toDfsEndpointUrl(extractedCreds.url) + "?" + extractedCreds.accountSas, pipeline);
        }
        else {
            throw new Error("Connection string must be either an Account connection string or a SAS connection string");
        }
    }
    constructor(url, credentialOrPipeline, 
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-options */
    options) {
        if (isPipelineLike(credentialOrPipeline)) {
            super(url, credentialOrPipeline);
        }
        else {
            let credential;
            if (credentialOrPipeline === undefined) {
                credential = new AnonymousCredential();
            }
            else {
                credential = credentialOrPipeline;
            }
            const pipeline = newPipeline(credential, options);
            super(url, pipeline);
        }
        // this.serviceContext = new Service(this.storageClientContext);
        this.blobServiceClient = new BlobServiceClient(this.blobEndpointUrl, this.pipeline);
    }
    /**
     * Creates a {@link DataLakeFileSystemClient} object.
     *
     * @param fileSystemName - File system name.
     */
    // Legacy, no way to fix the eslint error without breaking. Disable the rule for this line.
    /* eslint-disable-next-line @azure/azure-sdk/ts-naming-subclients */
    getFileSystemClient(fileSystemName) {
        return new DataLakeFileSystemClient(appendToURLPath(this.url, encodeURIComponent(fileSystemName)), this.pipeline);
    }
    /**
     * ONLY AVAILABLE WHEN USING BEARER TOKEN AUTHENTICATION (TokenCredential).
     *
     * Retrieves a user delegation key for the Data Lake service. This is only a valid operation when using
     * bearer token authentication.
     *
     * @example
     * ```js
     * // Generate user delegation SAS for a file system
     * const userDelegationKey = await dataLakeServiceClient.getUserDelegationKey(startsOn, expiresOn);
     * const fileSystemSAS = generateDataLakeSASQueryParameters({
     *     fileSystemName, // Required
     *     permissions: FileSystemSASPermissions.parse("racwdl"), // Required
     *     startsOn, // Required. Date type
     *     expiresOn, // Optional. Date type
     *     ipRange: { start: "0.0.0.0", end: "255.255.255.255" }, // Optional
     *     protocol: SASProtocol.HttpsAndHttp, // Optional
     *     version: "2018-11-09" // Must greater than or equal to 2018-11-09 to generate user delegation SAS
     *   },
     *   userDelegationKey, // UserDelegationKey
     *   accountName
     * ).toString();
     * ```
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/get-user-delegation-key
     *
     * @param startsOn - The start time for the user delegation SAS. Must be within 7 days of the current time.
     * @param expiresOn - The end time for the user delegation SAS. Must be within 7 days of the current time.
     * @param options -
     */
    async getUserDelegationKey(startsOn, expiresOn, options = {}) {
        return tracingClient.withSpan("DataLakeServiceClient-getUserDelegationKey", options, async (updatedOptions) => {
            return this.blobServiceClient.getUserDelegationKey(startsOn, expiresOn, updatedOptions);
        });
    }
    /**
     * Returns an async iterable iterator to list all the file systems
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the file systems in pages.
     *
     * Example using `for await` syntax:
     *
     * ```js
     * let i = 1;
     * for await (const fileSystem of serviceClient.listFileSystems()) {
     *   console.log(`FileSystem ${i++}: ${fileSystem.name}`);
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```js
     * let i = 1;
     * const iter = serviceClient.listFileSystems();
     * let fileSystemItem = await iter.next();
     * while (!fileSystemItem.done) {
     *   console.log(`FileSystem ${i++}: ${fileSystemItem.value.name}`);
     *   fileSystemItem = await iter.next();
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```js
     * // passing optional maxPageSize in the page settings
     * let i = 1;
     * for await (const response of serviceClient.listFileSystems().byPage({ maxPageSize: 20 })) {
     *   if (response.fileSystemItems) {
     *     for (const fileSystem of response.fileSystemItems) {
     *       console.log(`FileSystem ${i++}: ${fileSystem.name}`);
     *     }
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```js
     * let i = 1;
     * let iterator = serviceClient.listFileSystems().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * // Prints 2 file system names
     * if (response.fileSystemItems) {
     *   for (const fileSystem of response.fileSystemItems) {
     *     console.log(`FileSystem ${i++}: ${fileSystem.name}`);
     *   }
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     * // Passing next marker as continuationToken
     * iterator = serviceClient
     *   .listContainers()
     *   .byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * // Prints 10 file system names
     * if (response.fileSystemItems) {
     *   for (const fileSystem of response.fileSystemItems) {
     *      console.log(`FileSystem ${i++}: ${fileSystem.name}`);
     *   }
     * }
     * ```
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/list-containers2
     *
     * @param options -
     */
    listFileSystems(options = {}) {
        return toFileSystemPagedAsyncIterableIterator(this.blobServiceClient.listContainers(options));
    }
    // public async createFileSystem(): Promise<DataLakeFileSystemClient> {
    //   throw Error("NotImplemented");
    // }
    // public async deleteFileSystem(fileSystem: string): Promise<ServiceDeleteFileSystemResponse> {
    //   throw Error("NotImplemented");
    // }
    /**
     * Only available for DataLakeServiceClient constructed with a shared key credential.
     *
     * Generates an account Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/create-account-sas
     *
     * @param expiresOn - Optional. The time at which the shared access signature becomes invalid. Default to an hour later if not specified.
     * @param permissions - Specifies the list of permissions to be associated with the SAS.
     * @param resourceTypes - Specifies the resource types associated with the shared access signature.
     * @param options - Optional parameters.
     * @returns An account SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateAccountSasUrl(expiresOn, permissions = AccountSASPermissions.parse("r"), resourceTypes = "sco", options = {}) {
        if (!(this.credential instanceof StorageSharedKeyCredential)) {
            throw RangeError("Can only generate the account SAS when the client is initialized with a shared key credential");
        }
        if (expiresOn === undefined) {
            const now = new Date();
            expiresOn = new Date(now.getTime() + 3600 * 1000);
        }
        const sas = generateAccountSASQueryParameters(Object.assign({ permissions,
            expiresOn,
            resourceTypes, services: AccountSASServices.parse("b").toString() }, options), this.credential).toString();
        return appendToURLQuery(this.url, sas);
    }
    /**
     * Only available for DataLakeServiceClient constructed with a shared key credential.
     *
     * Generates string to sign for an account Shared Access Signature (SAS) based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/create-account-sas
     *
     * @param expiresOn - Optional. The time at which the shared access signature becomes invalid. Default to an hour later if not specified.
     * @param permissions - Specifies the list of permissions to be associated with the SAS.
     * @param resourceTypes - Specifies the resource types associated with the shared access signature.
     * @param options - Optional parameters.
     * @returns An account SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasStringToSign(expiresOn, permissions = AccountSASPermissions.parse("r"), resourceTypes = "sco", options = {}) {
        if (!(this.credential instanceof StorageSharedKeyCredential)) {
            throw RangeError("Can only generate the account SAS when the client is initialized with a shared key credential");
        }
        if (expiresOn === undefined) {
            const now = new Date();
            expiresOn = new Date(now.getTime() + 3600 * 1000);
        }
        return generateAccountSASQueryParametersInternal(Object.assign({ permissions,
            expiresOn,
            resourceTypes, services: AccountSASServices.parse("b").toString() }, options), this.credential).stringToSign;
    }
    /**
     * Renames an existing File System.
     *
     * @param sourceFileSystemName - The name of the source File System.
     * @param destinationContainerName - The new name of the File System.
     * @param options - Options to configure File System Rename operation.
     */
    /* eslint-disable-next-line @typescript-eslint/ban-ts-comment */
    // @ts-ignore Need to hide this interface for now. Make it public and turn on the live tests for it when the service is ready.
    async renameFileSystem(sourceFileSystemName, destinationFileSystemName, options = {}) {
        return tracingClient.withSpan("DataLakeServiceClient-renameFileSystem", options, async (updatedOptions) => {
            const res = await this.blobServiceClient["renameContainer"](sourceFileSystemName, destinationFileSystemName, updatedOptions);
            const fileSystemClient = this.getFileSystemClient(destinationFileSystemName);
            return {
                fileSystemClient,
                fileSystemRenameResponse: res.containerRenameResponse,
            };
        });
    }
    /**
     * Restore a previously deleted File System.
     * This API is only functional if Container Soft Delete is enabled for the storage account.
     *
     * @param deletedFileSystemName - The name of the source File System.
     * @param deleteFileSystemVersion - The new name of the File System.
     * @param options - Options to configure File System Restore operation.
     */
    async undeleteFileSystem(deletedFileSystemName, deleteFileSystemVersion, options = {}) {
        return tracingClient.withSpan("DataLakeServiceClient-undeleteFileSystem", options, async (updatedOptions) => {
            const res = await this.blobServiceClient.undeleteContainer(deletedFileSystemName, deleteFileSystemVersion, Object.assign(Object.assign({}, options), { destinationContainerName: options.destinationFileSystemName, tracingOptions: updatedOptions.tracingOptions }));
            const fileSystemClient = this.getFileSystemClient(options.destinationFileSystemName || deletedFileSystemName);
            return {
                fileSystemClient,
                fileSystemUndeleteResponse: res.containerUndeleteResponse,
            };
        });
    }
    /**
     * Gets the properties of a storage account’s Blob service endpoint, including properties
     * for Storage Analytics and CORS (Cross-Origin Resource Sharing) rules.
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/get-blob-service-properties
     *
     * @param options - Options to the Service Get Properties operation.
     * @returns Response data for the Service Get Properties operation.
     */
    async getProperties(options = {}) {
        return tracingClient.withSpan("DataLakeServiceClient-getProperties", options, async (updatedOptions) => {
            return this.blobServiceClient.getProperties({
                abortSignal: options.abortSignal,
                tracingOptions: updatedOptions.tracingOptions,
            });
        });
    }
    /**
     * Sets properties for a storage account’s Blob service endpoint, including properties
     * for Storage Analytics, CORS (Cross-Origin Resource Sharing) rules and soft delete settings.
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/set-blob-service-properties
     *
     * @param properties -
     * @param options - Options to the Service Set Properties operation.
     * @returns Response data for the Service Set Properties operation.
     */
    async setProperties(properties, options = {}) {
        return tracingClient.withSpan("DataLakeServiceClient-setProperties", options, async (updatedOptions) => {
            return this.blobServiceClient.setProperties(properties, {
                abortSignal: options.abortSignal,
                tracingOptions: updatedOptions.tracingOptions,
            });
        });
    }
}
//# sourceMappingURL=DataLakeServiceClient.js.map