import { __rest } from "tslib";
import { StorageContextClient } from "./StorageContextClient";
import { BlobServiceClient } from "@azure/storage-blob";
import { toBlobEndpointUrl, toDfsEndpointUrl } from "./transforms";
import { escapeURLPath, getAccountNameFromUrl, getURLScheme, iEqual } from "./utils/utils.common";
// This function relies on the Pipeline already being initialized by a storage-blob client
function getCoreClientOptions(pipeline) {
    const _a = pipeline.options, { httpClient: v1Client } = _a, restOptions = __rest(_a, ["httpClient"]);
    const httpClient = pipeline._coreHttpClient;
    if (!httpClient) {
        throw new Error("Pipeline not correctly initialized; missing V2 HttpClient");
    }
    const corePipeline = pipeline._corePipeline;
    if (!corePipeline) {
        throw new Error("Pipeline not correctly initialized; missing V2 Pipeline");
    }
    return Object.assign(Object.assign({}, restOptions), { allowInsecureConnection: true, httpClient, pipeline: corePipeline });
}
/**
 * A StorageClient represents a based URL class for {@link BlobServiceClient}, {@link ContainerClient}
 * and etc.
 */
export class StorageClient {
    /**
     * Creates an instance of StorageClient.
     * @param url - url to resource
     * @param pipeline - request policy pipeline.
     */
    constructor(url, pipeline) {
        // URL should be encoded and only once, protocol layer shouldn't encode URL again
        this.url = escapeURLPath(url);
        this.blobEndpointUrl = toBlobEndpointUrl(this.url);
        this.dfsEndpointUrl = toDfsEndpointUrl(this.url);
        this.accountName = getAccountNameFromUrl(this.blobEndpointUrl);
        this.pipeline = pipeline;
        // creating this BlobServiceClient allows us to use the converted V2 Pipeline attached to `pipeline`.
        const blobClient = new BlobServiceClient(url, pipeline);
        this.storageClientContext = new StorageContextClient(this.dfsEndpointUrl, getCoreClientOptions(pipeline));
        this.storageClientContextToBlobEndpoint = new StorageContextClient(this.blobEndpointUrl, getCoreClientOptions(pipeline));
        this.isHttps = iEqual(getURLScheme(this.url) || "", "https");
        this.credential = blobClient.credential;
        // Override protocol layer's default content-type
        const storageClientContext = this.storageClientContext;
        storageClientContext.requestContentType = undefined;
        const storageClientContextWithBlobEndpoint = this.storageClientContextToBlobEndpoint;
        storageClientContextWithBlobEndpoint.requestContentType = undefined;
    }
}
//# sourceMappingURL=StorageClient.js.map