// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * This is a helper class to construct a string representing the permissions granted by a ServiceSAS. Setting
 * a value to true means that any SAS which uses these permissions will grant permissions for that operation. Once all
 * the values are set, this should be serialized with toString and set as the permissions field on a
 * {@link DataLakeSASSignatureValues} object. It is possible to construct the permissions string without this class, but
 * the order of the permissions is particular and this class guarantees correctness.
 */
export class DataLakeSASPermissions {
    constructor() {
        /**
         * Specifies Read access granted.
         */
        this.read = false;
        /**
         * Specifies Add access granted.
         */
        this.add = false;
        /**
         * Specifies Create access granted.
         */
        this.create = false;
        /**
         * Specifies Write access granted.
         */
        this.write = false;
        /**
         * Specifies Delete access granted.
         */
        this.delete = false;
        /**
         * Specifies Move access granted.
         */
        this.move = false;
        /**
         * Specifies Execute access granted.
         */
        this.execute = false;
        /**
         * Specifies Ownership access granted, which allows the caller to set owner, owning group,
         * or act as the owner when renaming or deleting a blob (file or directory) within a folder
         * that has the sticky bit set.
         */
        this.manageOwnership = false;
        /**
         * Specifies Permission access granted, which allows the caller to set permissions and
         * POSIX ACLs on blobs (files and directories).
         */
        this.manageAccessControl = false;
    }
    /**
     * Creates a {@link DataLakeSASPermissions} from the specified permissions string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid permission.
     *
     * @param permissions -
     */
    static parse(permissions) {
        const blobSASPermissions = new DataLakeSASPermissions();
        for (const char of permissions) {
            switch (char) {
                case "r":
                    blobSASPermissions.read = true;
                    break;
                case "a":
                    blobSASPermissions.add = true;
                    break;
                case "c":
                    blobSASPermissions.create = true;
                    break;
                case "w":
                    blobSASPermissions.write = true;
                    break;
                case "d":
                    blobSASPermissions.delete = true;
                    break;
                case "m":
                    blobSASPermissions.move = true;
                    break;
                case "e":
                    blobSASPermissions.execute = true;
                    break;
                case "o":
                    blobSASPermissions.manageOwnership = true;
                    break;
                case "p":
                    blobSASPermissions.manageAccessControl = true;
                    break;
                default:
                    throw new RangeError(`Invalid permission: ${char}`);
            }
        }
        return blobSASPermissions;
    }
    /**
     * Converts the given permissions to a string. Using this method will guarantee the permissions are in an
     * order accepted by the service.
     *
     * @returns A string which represents the DataLakeSASPermissions
     */
    toString() {
        const permissions = [];
        if (this.read) {
            permissions.push("r");
        }
        if (this.add) {
            permissions.push("a");
        }
        if (this.create) {
            permissions.push("c");
        }
        if (this.write) {
            permissions.push("w");
        }
        if (this.delete) {
            permissions.push("d");
        }
        if (this.move) {
            permissions.push("m");
        }
        if (this.execute) {
            permissions.push("e");
        }
        if (this.manageOwnership) {
            permissions.push("o");
        }
        if (this.manageAccessControl) {
            permissions.push("p");
        }
        return permissions.join("");
    }
}
//# sourceMappingURL=DataLakeSASPermissions.js.map