// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
import { StorageSharedKeyCredential } from "../credentials/StorageSharedKeyCredential";
import { UserDelegationKeyCredential } from "../credentials/UserDelegationKeyCredential";
import { DataLakeSASPermissions } from "./DataLakeSASPermissions";
import { FileSystemSASPermissions } from "./FileSystemSASPermissions";
import { ipRangeToString } from "./SasIPRange";
import { SASQueryParameters } from "./SASQueryParameters";
import { SERVICE_VERSION } from "../utils/constants";
import { truncatedISO8061Date } from "../utils/utils.common";
import { DirectorySASPermissions } from "./DirectorySASPermissions";
export function generateDataLakeSASQueryParameters(dataLakeSASSignatureValues, sharedKeyCredentialOrUserDelegationKey, accountName) {
    return generateDataLakeSASQueryParametersInternal(dataLakeSASSignatureValues, sharedKeyCredentialOrUserDelegationKey, accountName).sasQueryParameter;
}
export function generateDataLakeSASQueryParametersInternal(dataLakeSASSignatureValues, sharedKeyCredentialOrUserDelegationKey, accountName) {
    const version = dataLakeSASSignatureValues.version
        ? dataLakeSASSignatureValues.version
        : SERVICE_VERSION;
    const sharedKeyCredential = sharedKeyCredentialOrUserDelegationKey instanceof StorageSharedKeyCredential
        ? sharedKeyCredentialOrUserDelegationKey
        : undefined;
    let userDelegationKeyCredential;
    if (sharedKeyCredential === undefined && accountName !== undefined) {
        userDelegationKeyCredential = new UserDelegationKeyCredential(accountName, sharedKeyCredentialOrUserDelegationKey);
    }
    if (sharedKeyCredential === undefined && userDelegationKeyCredential === undefined) {
        throw TypeError("Invalid sharedKeyCredential, userDelegationKey or accountName.");
    }
    // Version 2020-12-06 adds support for encryptionscope in SAS.
    if (version >= "2020-12-06") {
        if (sharedKeyCredential !== undefined) {
            return generateBlobSASQueryParameters20201206(dataLakeSASSignatureValues, sharedKeyCredential);
        }
        else {
            return generateBlobSASQueryParametersUDK20201206(dataLakeSASSignatureValues, userDelegationKeyCredential);
        }
    }
    // Version 2018-11-09 adds support for the signed resource and signed blob snapshot time fields.
    // https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas#constructing-the-signature-string
    if (version >= "2018-11-09") {
        if (sharedKeyCredential !== undefined) {
            return generateBlobSASQueryParameters20181109(dataLakeSASSignatureValues, sharedKeyCredential);
        }
        else {
            // Version 2020-02-10 delegation SAS signature construction includes preauthorizedAgentObjectId, agentObjectId, correlationId.
            if (version >= "2020-02-10") {
                return generateBlobSASQueryParametersUDK20200210(dataLakeSASSignatureValues, userDelegationKeyCredential);
            }
            else {
                return generateBlobSASQueryParametersUDK20181109(dataLakeSASSignatureValues, userDelegationKeyCredential);
            }
        }
    }
    if (version >= "2015-04-05") {
        if (sharedKeyCredential !== undefined) {
            return generateBlobSASQueryParameters20150405(dataLakeSASSignatureValues, sharedKeyCredential);
        }
        else {
            throw new RangeError("'version' must be >= '2018-11-09' when generating user delegation SAS using user delegation key.");
        }
    }
    throw new RangeError("'version' must be >= '2015-04-05'.");
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 * IMPLEMENTATION FOR API VERSION FROM 2015-04-05 AND BEFORE 2018-11-09.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startsOn and identifier.
 *
 * WARNING: When identifier is not provided, permissions and expiresOn are required.
 * You MUST assign value to identifier or expiresOn & permissions manually if you initial with
 * this constructor.
 *
 * @param dataLakeSASSignatureValues -
 * @param sharedKeyCredential -
 */
function generateBlobSASQueryParameters20150405(dataLakeSASSignatureValues, sharedKeyCredential) {
    if (!dataLakeSASSignatureValues.identifier &&
        !(dataLakeSASSignatureValues.permissions && dataLakeSASSignatureValues.expiresOn)) {
        throw new RangeError("Must provide 'permissions' and 'expiresOn' for DataLake SAS generation when 'identifier' is not provided.");
    }
    const version = dataLakeSASSignatureValues.version
        ? dataLakeSASSignatureValues.version
        : SERVICE_VERSION;
    dataLakeSASSignatureValues = SASSignatureValuesSanityCheckAndAutofill(dataLakeSASSignatureValues, version);
    let resource = "c";
    if (dataLakeSASSignatureValues.pathName) {
        resource = "b";
    }
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    let verifiedPermissions;
    if (dataLakeSASSignatureValues.permissions) {
        if (dataLakeSASSignatureValues.pathName) {
            verifiedPermissions = DataLakeSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
        }
        else {
            verifiedPermissions = FileSystemSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    const stringToSign = [
        verifiedPermissions ? verifiedPermissions : "",
        dataLakeSASSignatureValues.startsOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.startsOn, false)
            : "",
        dataLakeSASSignatureValues.expiresOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.expiresOn, false)
            : "",
        getCanonicalName(sharedKeyCredential.accountName, dataLakeSASSignatureValues.fileSystemName, dataLakeSASSignatureValues.pathName),
        dataLakeSASSignatureValues.identifier,
        dataLakeSASSignatureValues.ipRange ? ipRangeToString(dataLakeSASSignatureValues.ipRange) : "",
        dataLakeSASSignatureValues.protocol ? dataLakeSASSignatureValues.protocol : "",
        version,
        dataLakeSASSignatureValues.cacheControl ? dataLakeSASSignatureValues.cacheControl : "",
        dataLakeSASSignatureValues.contentDisposition
            ? dataLakeSASSignatureValues.contentDisposition
            : "",
        dataLakeSASSignatureValues.contentEncoding ? dataLakeSASSignatureValues.contentEncoding : "",
        dataLakeSASSignatureValues.contentLanguage ? dataLakeSASSignatureValues.contentLanguage : "",
        dataLakeSASSignatureValues.contentType ? dataLakeSASSignatureValues.contentType : "",
    ].join("\n");
    const signature = sharedKeyCredential.computeHMACSHA256(stringToSign);
    return {
        sasQueryParameter: new SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, dataLakeSASSignatureValues.protocol, dataLakeSASSignatureValues.startsOn, dataLakeSASSignatureValues.expiresOn, dataLakeSASSignatureValues.ipRange, dataLakeSASSignatureValues.identifier, resource, dataLakeSASSignatureValues.cacheControl, dataLakeSASSignatureValues.contentDisposition, dataLakeSASSignatureValues.contentEncoding, dataLakeSASSignatureValues.contentLanguage, dataLakeSASSignatureValues.contentType),
        stringToSign: stringToSign,
    };
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 * IMPLEMENTATION FOR API VERSION FROM 2018-11-09.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startsOn and identifier.
 *
 * WARNING: When identifier is not provided, permissions and expiresOn are required.
 * You MUST assign value to identifier or expiresOn & permissions manually if you initial with
 * this constructor.
 *
 * @param dataLakeSASSignatureValues -
 * @param sharedKeyCredential -
 */
function generateBlobSASQueryParameters20181109(dataLakeSASSignatureValues, sharedKeyCredential) {
    if (!dataLakeSASSignatureValues.identifier &&
        !(dataLakeSASSignatureValues.permissions && dataLakeSASSignatureValues.expiresOn)) {
        throw new RangeError("Must provide 'permissions' and 'expiresOn' for Blob SAS generation when 'identifier' is not provided.");
    }
    const version = dataLakeSASSignatureValues.version
        ? dataLakeSASSignatureValues.version
        : SERVICE_VERSION;
    dataLakeSASSignatureValues = SASSignatureValuesSanityCheckAndAutofill(dataLakeSASSignatureValues, version);
    let resource = "c";
    if (dataLakeSASSignatureValues.pathName) {
        if (dataLakeSASSignatureValues.isDirectory) {
            resource = "d";
        }
        else {
            resource = "b";
            if (dataLakeSASSignatureValues.snapshotTime) {
                resource = "bs";
            }
        }
    }
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    let verifiedPermissions;
    if (dataLakeSASSignatureValues.permissions) {
        if (dataLakeSASSignatureValues.pathName) {
            if (dataLakeSASSignatureValues.isDirectory) {
                verifiedPermissions = DirectorySASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
            else {
                verifiedPermissions = DataLakeSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
        }
        else {
            verifiedPermissions = FileSystemSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    const stringToSign = [
        verifiedPermissions ? verifiedPermissions : "",
        dataLakeSASSignatureValues.startsOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.startsOn, false)
            : "",
        dataLakeSASSignatureValues.expiresOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.expiresOn, false)
            : "",
        getCanonicalName(sharedKeyCredential.accountName, dataLakeSASSignatureValues.fileSystemName, dataLakeSASSignatureValues.pathName),
        dataLakeSASSignatureValues.identifier,
        dataLakeSASSignatureValues.ipRange ? ipRangeToString(dataLakeSASSignatureValues.ipRange) : "",
        dataLakeSASSignatureValues.protocol ? dataLakeSASSignatureValues.protocol : "",
        version,
        resource,
        dataLakeSASSignatureValues.snapshotTime,
        dataLakeSASSignatureValues.cacheControl ? dataLakeSASSignatureValues.cacheControl : "",
        dataLakeSASSignatureValues.contentDisposition
            ? dataLakeSASSignatureValues.contentDisposition
            : "",
        dataLakeSASSignatureValues.contentEncoding ? dataLakeSASSignatureValues.contentEncoding : "",
        dataLakeSASSignatureValues.contentLanguage ? dataLakeSASSignatureValues.contentLanguage : "",
        dataLakeSASSignatureValues.contentType ? dataLakeSASSignatureValues.contentType : "",
    ].join("\n");
    const signature = sharedKeyCredential.computeHMACSHA256(stringToSign);
    return {
        sasQueryParameter: new SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, dataLakeSASSignatureValues.protocol, dataLakeSASSignatureValues.startsOn, dataLakeSASSignatureValues.expiresOn, dataLakeSASSignatureValues.ipRange, dataLakeSASSignatureValues.identifier, resource, dataLakeSASSignatureValues.cacheControl, dataLakeSASSignatureValues.contentDisposition, dataLakeSASSignatureValues.contentEncoding, dataLakeSASSignatureValues.contentLanguage, dataLakeSASSignatureValues.contentType, undefined, dataLakeSASSignatureValues.directoryDepth),
        stringToSign: stringToSign,
    };
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 * IMPLEMENTATION FOR API VERSION FROM 2018-11-09.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startsOn.
 *
 * WARNING: identifier will be ignored, permissions and expiresOn are required.
 *
 * @param dataLakeSASSignatureValues -
 * @param userDelegationKeyCredential -
 */
function generateBlobSASQueryParametersUDK20181109(dataLakeSASSignatureValues, userDelegationKeyCredential) {
    if (!dataLakeSASSignatureValues.permissions || !dataLakeSASSignatureValues.expiresOn) {
        throw new RangeError("Must provide 'permissions' and 'expiresOn' for Blob SAS generation when generating user delegation SAS.");
    }
    const version = dataLakeSASSignatureValues.version
        ? dataLakeSASSignatureValues.version
        : SERVICE_VERSION;
    dataLakeSASSignatureValues = SASSignatureValuesSanityCheckAndAutofill(dataLakeSASSignatureValues, version);
    let resource = "c";
    if (dataLakeSASSignatureValues.pathName) {
        if (dataLakeSASSignatureValues.isDirectory) {
            resource = "d";
        }
        else {
            resource = "b";
            if (dataLakeSASSignatureValues.snapshotTime) {
                resource = "bs";
            }
        }
    }
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    let verifiedPermissions;
    if (dataLakeSASSignatureValues.permissions) {
        if (dataLakeSASSignatureValues.pathName) {
            if (dataLakeSASSignatureValues.isDirectory) {
                verifiedPermissions = DirectorySASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
            else {
                verifiedPermissions = DataLakeSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
        }
        else {
            verifiedPermissions = FileSystemSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    const stringToSign = [
        verifiedPermissions ? verifiedPermissions : "",
        dataLakeSASSignatureValues.startsOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.startsOn, false)
            : "",
        dataLakeSASSignatureValues.expiresOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.expiresOn, false)
            : "",
        getCanonicalName(userDelegationKeyCredential.accountName, dataLakeSASSignatureValues.fileSystemName, dataLakeSASSignatureValues.pathName),
        userDelegationKeyCredential.userDelegationKey.signedObjectId,
        userDelegationKeyCredential.userDelegationKey.signedTenantId,
        userDelegationKeyCredential.userDelegationKey.signedStartsOn
            ? truncatedISO8061Date(userDelegationKeyCredential.userDelegationKey.signedStartsOn, false)
            : "",
        userDelegationKeyCredential.userDelegationKey.signedExpiresOn
            ? truncatedISO8061Date(userDelegationKeyCredential.userDelegationKey.signedExpiresOn, false)
            : "",
        userDelegationKeyCredential.userDelegationKey.signedService,
        userDelegationKeyCredential.userDelegationKey.signedVersion,
        dataLakeSASSignatureValues.ipRange ? ipRangeToString(dataLakeSASSignatureValues.ipRange) : "",
        dataLakeSASSignatureValues.protocol ? dataLakeSASSignatureValues.protocol : "",
        version,
        resource,
        dataLakeSASSignatureValues.snapshotTime,
        dataLakeSASSignatureValues.cacheControl,
        dataLakeSASSignatureValues.contentDisposition,
        dataLakeSASSignatureValues.contentEncoding,
        dataLakeSASSignatureValues.contentLanguage,
        dataLakeSASSignatureValues.contentType,
    ].join("\n");
    const signature = userDelegationKeyCredential.computeHMACSHA256(stringToSign);
    return {
        sasQueryParameter: new SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, dataLakeSASSignatureValues.protocol, dataLakeSASSignatureValues.startsOn, dataLakeSASSignatureValues.expiresOn, dataLakeSASSignatureValues.ipRange, dataLakeSASSignatureValues.identifier, resource, dataLakeSASSignatureValues.cacheControl, dataLakeSASSignatureValues.contentDisposition, dataLakeSASSignatureValues.contentEncoding, dataLakeSASSignatureValues.contentLanguage, dataLakeSASSignatureValues.contentType, userDelegationKeyCredential.userDelegationKey, dataLakeSASSignatureValues.directoryDepth, dataLakeSASSignatureValues.preauthorizedAgentObjectId, dataLakeSASSignatureValues.agentObjectId, dataLakeSASSignatureValues.correlationId),
        stringToSign: stringToSign,
    };
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 * IMPLEMENTATION FOR API VERSION FROM 2020-02-10.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startsOn.
 *
 * WARNING: identifier will be ignored, permissions and expiresOn are required.
 *
 * @param dataLakeSASSignatureValues -
 * @param userDelegationKeyCredential -
 */
function generateBlobSASQueryParametersUDK20200210(dataLakeSASSignatureValues, userDelegationKeyCredential) {
    if (!dataLakeSASSignatureValues.permissions || !dataLakeSASSignatureValues.expiresOn) {
        throw new RangeError("Must provide 'permissions' and 'expiresOn' for Blob SAS generation when generating user delegation SAS.");
    }
    const version = dataLakeSASSignatureValues.version
        ? dataLakeSASSignatureValues.version
        : SERVICE_VERSION;
    dataLakeSASSignatureValues = SASSignatureValuesSanityCheckAndAutofill(dataLakeSASSignatureValues, version);
    let resource = "c";
    if (dataLakeSASSignatureValues.pathName) {
        if (dataLakeSASSignatureValues.isDirectory) {
            resource = "d";
        }
        else {
            resource = "b";
            if (dataLakeSASSignatureValues.snapshotTime) {
                resource = "bs";
            }
        }
    }
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    let verifiedPermissions;
    if (dataLakeSASSignatureValues.permissions) {
        if (dataLakeSASSignatureValues.pathName) {
            if (dataLakeSASSignatureValues.isDirectory) {
                verifiedPermissions = DirectorySASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
            else {
                verifiedPermissions = DataLakeSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
        }
        else {
            verifiedPermissions = FileSystemSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    const stringToSign = [
        verifiedPermissions ? verifiedPermissions : "",
        dataLakeSASSignatureValues.startsOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.startsOn, false)
            : "",
        dataLakeSASSignatureValues.expiresOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.expiresOn, false)
            : "",
        getCanonicalName(userDelegationKeyCredential.accountName, dataLakeSASSignatureValues.fileSystemName, dataLakeSASSignatureValues.pathName),
        userDelegationKeyCredential.userDelegationKey.signedObjectId,
        userDelegationKeyCredential.userDelegationKey.signedTenantId,
        userDelegationKeyCredential.userDelegationKey.signedStartsOn
            ? truncatedISO8061Date(userDelegationKeyCredential.userDelegationKey.signedStartsOn, false)
            : "",
        userDelegationKeyCredential.userDelegationKey.signedExpiresOn
            ? truncatedISO8061Date(userDelegationKeyCredential.userDelegationKey.signedExpiresOn, false)
            : "",
        userDelegationKeyCredential.userDelegationKey.signedService,
        userDelegationKeyCredential.userDelegationKey.signedVersion,
        dataLakeSASSignatureValues.preauthorizedAgentObjectId,
        dataLakeSASSignatureValues.agentObjectId,
        dataLakeSASSignatureValues.correlationId,
        dataLakeSASSignatureValues.ipRange ? ipRangeToString(dataLakeSASSignatureValues.ipRange) : "",
        dataLakeSASSignatureValues.protocol ? dataLakeSASSignatureValues.protocol : "",
        version,
        resource,
        dataLakeSASSignatureValues.snapshotTime,
        dataLakeSASSignatureValues.cacheControl,
        dataLakeSASSignatureValues.contentDisposition,
        dataLakeSASSignatureValues.contentEncoding,
        dataLakeSASSignatureValues.contentLanguage,
        dataLakeSASSignatureValues.contentType,
    ].join("\n");
    const signature = userDelegationKeyCredential.computeHMACSHA256(stringToSign);
    return {
        sasQueryParameter: new SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, dataLakeSASSignatureValues.protocol, dataLakeSASSignatureValues.startsOn, dataLakeSASSignatureValues.expiresOn, dataLakeSASSignatureValues.ipRange, dataLakeSASSignatureValues.identifier, resource, dataLakeSASSignatureValues.cacheControl, dataLakeSASSignatureValues.contentDisposition, dataLakeSASSignatureValues.contentEncoding, dataLakeSASSignatureValues.contentLanguage, dataLakeSASSignatureValues.contentType, userDelegationKeyCredential.userDelegationKey, dataLakeSASSignatureValues.directoryDepth, dataLakeSASSignatureValues.preauthorizedAgentObjectId, dataLakeSASSignatureValues.agentObjectId, dataLakeSASSignatureValues.correlationId),
        stringToSign: stringToSign,
    };
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 * IMPLEMENTATION FOR API VERSION FROM 2020-12-06.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startsOn and identifier.
 *
 * WARNING: When identifier is not provided, permissions and expiresOn are required.
 * You MUST assign value to identifier or expiresOn & permissions manually if you initial with
 * this constructor.
 *
 * @param dataLakeSASSignatureValues -
 * @param sharedKeyCredential -
 */
function generateBlobSASQueryParameters20201206(dataLakeSASSignatureValues, sharedKeyCredential) {
    if (!dataLakeSASSignatureValues.identifier &&
        !(dataLakeSASSignatureValues.permissions && dataLakeSASSignatureValues.expiresOn)) {
        throw new RangeError("Must provide 'permissions' and 'expiresOn' for Blob SAS generation when 'identifier' is not provided.");
    }
    const version = dataLakeSASSignatureValues.version
        ? dataLakeSASSignatureValues.version
        : SERVICE_VERSION;
    dataLakeSASSignatureValues = SASSignatureValuesSanityCheckAndAutofill(dataLakeSASSignatureValues, version);
    let resource = "c";
    if (dataLakeSASSignatureValues.pathName) {
        if (dataLakeSASSignatureValues.isDirectory) {
            resource = "d";
        }
        else {
            resource = "b";
            if (dataLakeSASSignatureValues.snapshotTime) {
                resource = "bs";
            }
        }
    }
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    let verifiedPermissions;
    if (dataLakeSASSignatureValues.permissions) {
        if (dataLakeSASSignatureValues.pathName) {
            if (dataLakeSASSignatureValues.isDirectory) {
                verifiedPermissions = DirectorySASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
            else {
                verifiedPermissions = DataLakeSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
        }
        else {
            verifiedPermissions = FileSystemSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    const stringToSign = [
        verifiedPermissions ? verifiedPermissions : "",
        dataLakeSASSignatureValues.startsOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.startsOn, false)
            : "",
        dataLakeSASSignatureValues.expiresOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.expiresOn, false)
            : "",
        getCanonicalName(sharedKeyCredential.accountName, dataLakeSASSignatureValues.fileSystemName, dataLakeSASSignatureValues.pathName),
        dataLakeSASSignatureValues.identifier,
        dataLakeSASSignatureValues.ipRange ? ipRangeToString(dataLakeSASSignatureValues.ipRange) : "",
        dataLakeSASSignatureValues.protocol ? dataLakeSASSignatureValues.protocol : "",
        version,
        resource,
        dataLakeSASSignatureValues.snapshotTime,
        dataLakeSASSignatureValues.encryptionScope ? dataLakeSASSignatureValues.encryptionScope : "",
        dataLakeSASSignatureValues.cacheControl ? dataLakeSASSignatureValues.cacheControl : "",
        dataLakeSASSignatureValues.contentDisposition
            ? dataLakeSASSignatureValues.contentDisposition
            : "",
        dataLakeSASSignatureValues.contentEncoding ? dataLakeSASSignatureValues.contentEncoding : "",
        dataLakeSASSignatureValues.contentLanguage ? dataLakeSASSignatureValues.contentLanguage : "",
        dataLakeSASSignatureValues.contentType ? dataLakeSASSignatureValues.contentType : "",
    ].join("\n");
    const signature = sharedKeyCredential.computeHMACSHA256(stringToSign);
    return {
        sasQueryParameter: new SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, dataLakeSASSignatureValues.protocol, dataLakeSASSignatureValues.startsOn, dataLakeSASSignatureValues.expiresOn, dataLakeSASSignatureValues.ipRange, dataLakeSASSignatureValues.identifier, resource, dataLakeSASSignatureValues.cacheControl, dataLakeSASSignatureValues.contentDisposition, dataLakeSASSignatureValues.contentEncoding, dataLakeSASSignatureValues.contentLanguage, dataLakeSASSignatureValues.contentType, undefined, dataLakeSASSignatureValues.directoryDepth, undefined, undefined, undefined, dataLakeSASSignatureValues.encryptionScope),
        stringToSign: stringToSign,
    };
}
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 * IMPLEMENTATION FOR API VERSION FROM 2020-12-06.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startsOn.
 *
 * WARNING: identifier will be ignored, permissions and expiresOn are required.
 *
 * @param dataLakeSASSignatureValues -
 * @param userDelegationKeyCredential -
 */
function generateBlobSASQueryParametersUDK20201206(dataLakeSASSignatureValues, userDelegationKeyCredential) {
    if (!dataLakeSASSignatureValues.permissions || !dataLakeSASSignatureValues.expiresOn) {
        throw new RangeError("Must provide 'permissions' and 'expiresOn' for Blob SAS generation when generating user delegation SAS.");
    }
    const version = dataLakeSASSignatureValues.version
        ? dataLakeSASSignatureValues.version
        : SERVICE_VERSION;
    dataLakeSASSignatureValues = SASSignatureValuesSanityCheckAndAutofill(dataLakeSASSignatureValues, version);
    let resource = "c";
    if (dataLakeSASSignatureValues.pathName) {
        if (dataLakeSASSignatureValues.isDirectory) {
            resource = "d";
        }
        else {
            resource = "b";
            if (dataLakeSASSignatureValues.snapshotTime) {
                resource = "bs";
            }
        }
    }
    // Calling parse and toString guarantees the proper ordering and throws on invalid characters.
    let verifiedPermissions;
    if (dataLakeSASSignatureValues.permissions) {
        if (dataLakeSASSignatureValues.pathName) {
            if (dataLakeSASSignatureValues.isDirectory) {
                verifiedPermissions = DirectorySASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
            else {
                verifiedPermissions = DataLakeSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
            }
        }
        else {
            verifiedPermissions = FileSystemSASPermissions.parse(dataLakeSASSignatureValues.permissions.toString()).toString();
        }
    }
    // Signature is generated on the un-url-encoded values.
    const stringToSign = [
        verifiedPermissions ? verifiedPermissions : "",
        dataLakeSASSignatureValues.startsOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.startsOn, false)
            : "",
        dataLakeSASSignatureValues.expiresOn
            ? truncatedISO8061Date(dataLakeSASSignatureValues.expiresOn, false)
            : "",
        getCanonicalName(userDelegationKeyCredential.accountName, dataLakeSASSignatureValues.fileSystemName, dataLakeSASSignatureValues.pathName),
        userDelegationKeyCredential.userDelegationKey.signedObjectId,
        userDelegationKeyCredential.userDelegationKey.signedTenantId,
        userDelegationKeyCredential.userDelegationKey.signedStartsOn
            ? truncatedISO8061Date(userDelegationKeyCredential.userDelegationKey.signedStartsOn, false)
            : "",
        userDelegationKeyCredential.userDelegationKey.signedExpiresOn
            ? truncatedISO8061Date(userDelegationKeyCredential.userDelegationKey.signedExpiresOn, false)
            : "",
        userDelegationKeyCredential.userDelegationKey.signedService,
        userDelegationKeyCredential.userDelegationKey.signedVersion,
        dataLakeSASSignatureValues.preauthorizedAgentObjectId,
        dataLakeSASSignatureValues.agentObjectId,
        dataLakeSASSignatureValues.correlationId,
        dataLakeSASSignatureValues.ipRange ? ipRangeToString(dataLakeSASSignatureValues.ipRange) : "",
        dataLakeSASSignatureValues.protocol ? dataLakeSASSignatureValues.protocol : "",
        version,
        resource,
        dataLakeSASSignatureValues.snapshotTime,
        dataLakeSASSignatureValues.encryptionScope,
        dataLakeSASSignatureValues.cacheControl,
        dataLakeSASSignatureValues.contentDisposition,
        dataLakeSASSignatureValues.contentEncoding,
        dataLakeSASSignatureValues.contentLanguage,
        dataLakeSASSignatureValues.contentType,
    ].join("\n");
    const signature = userDelegationKeyCredential.computeHMACSHA256(stringToSign);
    return {
        sasQueryParameter: new SASQueryParameters(version, signature, verifiedPermissions, undefined, undefined, dataLakeSASSignatureValues.protocol, dataLakeSASSignatureValues.startsOn, dataLakeSASSignatureValues.expiresOn, dataLakeSASSignatureValues.ipRange, dataLakeSASSignatureValues.identifier, resource, dataLakeSASSignatureValues.cacheControl, dataLakeSASSignatureValues.contentDisposition, dataLakeSASSignatureValues.contentEncoding, dataLakeSASSignatureValues.contentLanguage, dataLakeSASSignatureValues.contentType, userDelegationKeyCredential.userDelegationKey, dataLakeSASSignatureValues.directoryDepth, dataLakeSASSignatureValues.preauthorizedAgentObjectId, dataLakeSASSignatureValues.agentObjectId, dataLakeSASSignatureValues.correlationId, dataLakeSASSignatureValues.encryptionScope),
        stringToSign: stringToSign,
    };
}
function getCanonicalName(accountName, containerName, blobName) {
    // FileSystem: "/blob/account/fileSystemName"
    // File:       "/blob/account/fileSystemName/fileName"
    const elements = [`/blob/${accountName}/${containerName}`];
    if (blobName) {
        elements.push(`/${blobName}`);
    }
    return elements.join("");
}
function SASSignatureValuesSanityCheckAndAutofill(dataLakeSASSignatureValues, version) {
    var _a;
    if (version < "2020-02-10" &&
        (dataLakeSASSignatureValues.isDirectory || dataLakeSASSignatureValues.directoryDepth)) {
        throw RangeError("'version' must be >= '2020-02-10' to support directory SAS.");
    }
    if (dataLakeSASSignatureValues.isDirectory && dataLakeSASSignatureValues.pathName === undefined) {
        throw RangeError("Must provide 'pathName' when 'isDirectory' is true.");
    }
    if (dataLakeSASSignatureValues.directoryDepth !== undefined &&
        (!Number.isInteger(dataLakeSASSignatureValues.directoryDepth) ||
            dataLakeSASSignatureValues.directoryDepth < 0)) {
        throw RangeError("'directoryDepth' must be a non-negative interger.");
    }
    if (dataLakeSASSignatureValues.isDirectory &&
        dataLakeSASSignatureValues.directoryDepth === undefined) {
        // calculate directoryDepth from pathName
        if (dataLakeSASSignatureValues.pathName === "/") {
            dataLakeSASSignatureValues.directoryDepth = 0;
        }
        else {
            dataLakeSASSignatureValues.directoryDepth = (_a = dataLakeSASSignatureValues.pathName) === null || _a === void 0 ? void 0 : _a.split("/").filter((x) => x !== "").length;
        }
    }
    if (version < "2020-02-10" &&
        dataLakeSASSignatureValues.permissions &&
        (dataLakeSASSignatureValues.permissions.move ||
            dataLakeSASSignatureValues.permissions.execute ||
            dataLakeSASSignatureValues.permissions.manageOwnership ||
            dataLakeSASSignatureValues.permissions.manageAccessControl)) {
        throw RangeError("'version' must be >= '2020-02-10' when providing m, e, o or p permission.");
    }
    if (version < "2020-02-10" &&
        (dataLakeSASSignatureValues.preauthorizedAgentObjectId ||
            dataLakeSASSignatureValues.agentObjectId ||
            dataLakeSASSignatureValues.correlationId)) {
        throw RangeError("'version' must be >= '2020-02-10' when providing 'preauthorizedAgentObjectId', 'agentObjectId' or 'correlationId'.");
    }
    if (dataLakeSASSignatureValues.preauthorizedAgentObjectId &&
        dataLakeSASSignatureValues.agentObjectId) {
        throw RangeError("'preauthorizedAgentObjectId' or 'agentObjectId' shouldn't be specified at the same time.");
    }
    if (dataLakeSASSignatureValues.snapshotTime && version < "2018-11-09") {
        throw RangeError("'version' must be >= '2018-11-09' when provided 'snapshotTime'.");
    }
    if (dataLakeSASSignatureValues.pathName === undefined &&
        dataLakeSASSignatureValues.snapshotTime) {
        throw RangeError("Must provide 'blobName' when provided 'snapshotTime'.");
    }
    return dataLakeSASSignatureValues;
}
//# sourceMappingURL=DataLakeSASSignatureValues.js.map