import { ipRangeToString } from "./SasIPRange";
import { truncatedISO8061Date } from "../utils/utils.common";
/**
 * Protocols for generated SAS.
 */
export var SASProtocol;
(function (SASProtocol) {
    /**
     * Protocol that allows HTTPS only
     */
    SASProtocol["Https"] = "https";
    /**
     * Protocol that allows both HTTPS and HTTP
     */
    SASProtocol["HttpsAndHttp"] = "https,http";
})(SASProtocol || (SASProtocol = {}));
/**
 * Represents the components that make up an Azure Storage SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link AccountSASSignatureValues} and {@link BlobSASSignatureValues}
 * types. Once generated, it can be encoded into a {@link String} and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 */
export class SASQueryParameters {
    /**
     * Optional. IP range allowed for this SAS.
     *
     * @readonly
     */
    get ipRange() {
        if (this.ipRangeInner) {
            return {
                end: this.ipRangeInner.end,
                start: this.ipRangeInner.start,
            };
        }
        return undefined;
    }
    constructor(version, signature, permissionsOrOptions, services, resourceTypes, protocol, startsOn, expiresOn, ipRange, identifier, resource, cacheControl, contentDisposition, contentEncoding, contentLanguage, contentType, userDelegationKey, directoryDepth, preauthorizedAgentObjectId, agentObjectId, correlationId, encryptionScope) {
        this.version = version;
        this.signature = signature;
        if (permissionsOrOptions !== undefined && typeof permissionsOrOptions !== "string") {
            // SASQueryParametersOptions
            const options = permissionsOrOptions;
            this.services = options.services;
            this.resourceTypes = options.resourceTypes;
            this.expiresOn = options.expiresOn;
            this.permissions = options.permissions;
            this.protocol = options.protocol;
            this.startsOn = options.startsOn;
            this.ipRangeInner = options.ipRange;
            this.identifier = options.identifier;
            this.resource = options.resource;
            this.cacheControl = options.cacheControl;
            this.contentDisposition = options.contentDisposition;
            this.contentEncoding = options.contentEncoding;
            this.contentLanguage = options.contentLanguage;
            this.contentType = options.contentType;
            this.directoryDepth = options.directoryDepth;
            this.preauthorizedAgentObjectId = options.preauthorizedAgentObjectId;
            this.agentObjectId = options.agentObjectId;
            this.correlationId = options.correlationId;
            this.encryptionScope = options.encryptionScope;
            if (options.userDelegationKey) {
                this.signedOid = options.userDelegationKey.signedObjectId;
                this.signedTenantId = options.userDelegationKey.signedTenantId;
                this.signedStartsOn = options.userDelegationKey.signedStartsOn;
                this.signedExpiresOn = options.userDelegationKey.signedExpiresOn;
                this.signedService = options.userDelegationKey.signedService;
                this.signedVersion = options.userDelegationKey.signedVersion;
            }
        }
        else {
            this.services = services;
            this.resourceTypes = resourceTypes;
            this.expiresOn = expiresOn;
            this.permissions = permissionsOrOptions;
            this.protocol = protocol;
            this.startsOn = startsOn;
            this.ipRangeInner = ipRange;
            this.identifier = identifier;
            this.resource = resource;
            this.cacheControl = cacheControl;
            this.contentDisposition = contentDisposition;
            this.contentEncoding = contentEncoding;
            this.contentLanguage = contentLanguage;
            this.contentType = contentType;
            this.directoryDepth = directoryDepth;
            this.preauthorizedAgentObjectId = preauthorizedAgentObjectId;
            this.agentObjectId = agentObjectId;
            this.correlationId = correlationId;
            this.encryptionScope = encryptionScope;
            if (userDelegationKey) {
                this.signedOid = userDelegationKey.signedObjectId;
                this.signedTenantId = userDelegationKey.signedTenantId;
                this.signedStartsOn = userDelegationKey.signedStartsOn;
                this.signedExpiresOn = userDelegationKey.signedExpiresOn;
                this.signedService = userDelegationKey.signedService;
                this.signedVersion = userDelegationKey.signedVersion;
            }
        }
    }
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     */
    toString() {
        var _a;
        const params = [
            "sv",
            "ss",
            "srt",
            "spr",
            "st",
            "se",
            "sip",
            "si",
            "ses",
            "skoid", // Signed object ID
            "sktid", // Signed tenant ID
            "skt", // Signed key start time
            "ske", // Signed key expiry time
            "sks", // Signed key service
            "skv", // Signed key version
            "sr",
            "sp",
            "sig",
            "rscc",
            "rscd",
            "rsce",
            "rscl",
            "rsct",
            "sdd",
            "saoid",
            "suoid",
            "scid",
        ];
        const queries = [];
        for (const param of params) {
            switch (param) {
                case "sv":
                    this.tryAppendQueryParameter(queries, param, this.version);
                    break;
                case "ss":
                    this.tryAppendQueryParameter(queries, param, this.services);
                    break;
                case "srt":
                    this.tryAppendQueryParameter(queries, param, this.resourceTypes);
                    break;
                case "spr":
                    this.tryAppendQueryParameter(queries, param, this.protocol);
                    break;
                case "st":
                    this.tryAppendQueryParameter(queries, param, this.startsOn ? truncatedISO8061Date(this.startsOn, false) : undefined);
                    break;
                case "se":
                    this.tryAppendQueryParameter(queries, param, this.expiresOn ? truncatedISO8061Date(this.expiresOn, false) : undefined);
                    break;
                case "sip":
                    this.tryAppendQueryParameter(queries, param, this.ipRange ? ipRangeToString(this.ipRange) : undefined);
                    break;
                case "si":
                    this.tryAppendQueryParameter(queries, param, this.identifier);
                    break;
                case "ses":
                    this.tryAppendQueryParameter(queries, param, this.encryptionScope);
                    break;
                case "skoid": // Signed object ID
                    this.tryAppendQueryParameter(queries, param, this.signedOid);
                    break;
                case "sktid": // Signed tenant ID
                    this.tryAppendQueryParameter(queries, param, this.signedTenantId);
                    break;
                case "skt": // Signed key start time
                    this.tryAppendQueryParameter(queries, param, this.signedStartsOn ? truncatedISO8061Date(this.signedStartsOn, false) : undefined);
                    break;
                case "ske": // Signed key expiry time
                    this.tryAppendQueryParameter(queries, param, this.signedExpiresOn ? truncatedISO8061Date(this.signedExpiresOn, false) : undefined);
                    break;
                case "sks": // Signed key service
                    this.tryAppendQueryParameter(queries, param, this.signedService);
                    break;
                case "skv": // Signed key version
                    this.tryAppendQueryParameter(queries, param, this.signedVersion);
                    break;
                case "sr":
                    this.tryAppendQueryParameter(queries, param, this.resource);
                    break;
                case "sp":
                    this.tryAppendQueryParameter(queries, param, this.permissions);
                    break;
                case "sig":
                    this.tryAppendQueryParameter(queries, param, this.signature);
                    break;
                case "rscc":
                    this.tryAppendQueryParameter(queries, param, this.cacheControl);
                    break;
                case "rscd":
                    this.tryAppendQueryParameter(queries, param, this.contentDisposition);
                    break;
                case "rsce":
                    this.tryAppendQueryParameter(queries, param, this.contentEncoding);
                    break;
                case "rscl":
                    this.tryAppendQueryParameter(queries, param, this.contentLanguage);
                    break;
                case "rsct":
                    this.tryAppendQueryParameter(queries, param, this.contentType);
                    break;
                case "sdd":
                    this.tryAppendQueryParameter(queries, param, (_a = this.directoryDepth) === null || _a === void 0 ? void 0 : _a.toString());
                    break;
                case "saoid":
                    this.tryAppendQueryParameter(queries, param, this.preauthorizedAgentObjectId);
                    break;
                case "suoid":
                    this.tryAppendQueryParameter(queries, param, this.agentObjectId);
                    break;
                case "scid":
                    this.tryAppendQueryParameter(queries, param, this.correlationId);
                    break;
            }
        }
        return queries.join("&");
    }
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @param queries -
     * @param key -
     * @param value -
     */
    tryAppendQueryParameter(queries, key, value) {
        if (!value) {
            return;
        }
        key = encodeURIComponent(key);
        value = encodeURIComponent(value);
        if (key.length > 0 && value.length > 0) {
            queries.push(`${key}=${value}`);
        }
    }
}
//# sourceMappingURL=SASQueryParameters.js.map