import type { AbortSignalLike } from '@azure/abort-controller';
import { AnonymousCredential } from '@azure/storage-blob';
import { AnonymousCredentialPolicy } from '@azure/storage-blob';
import { AzureLogger } from '@azure/logger';
import { BaseRequestPolicy } from '@azure/storage-blob';
import type { BlobLeaseClient } from '@azure/storage-blob';
import type { BlobQueryArrowConfiguration } from '@azure/storage-blob';
import type { ContainerRenameResponse } from '@azure/storage-blob';
import type { ContainerUndeleteResponse } from '@azure/storage-blob';
import * as coreClient from '@azure/core-client';
import * as coreHttpCompat from '@azure/core-http-compat';
import * as coreRestPipeline from '@azure/core-rest-pipeline';
import { Credential as Credential_2 } from '@azure/storage-blob';
import { CredentialPolicy } from '@azure/storage-blob';
import { ServiceGetPropertiesResponse as DataLakeServiceGetPropertiesResponse } from '@azure/storage-blob';
import { BlobServiceProperties as DataLakeServiceProperties } from '@azure/storage-blob';
import { HttpHeadersLike as HttpHeaders } from '@azure/core-http-compat';
import { CompatResponse as HttpOperationResponse } from '@azure/core-http-compat';
import { RequestBodyType as HttpRequestBody } from '@azure/core-rest-pipeline';
import { isPipelineLike } from '@azure/storage-blob';
import type { KeepAliveOptions } from '@azure/core-http-compat';
import { Lease } from '@azure/storage-blob';
import { LeaseAccessConditions } from '@azure/storage-blob';
import { LeaseOperationOptions } from '@azure/storage-blob';
import { LeaseOperationResponse } from '@azure/storage-blob';
import type { ModifiedAccessConditions as ModifiedAccessConditions_3 } from '@azure/storage-blob';
import type { OperationTracingOptions } from '@azure/core-tracing';
import type { PagedAsyncIterableIterator } from '@azure/core-paging';
import { Pipeline } from '@azure/storage-blob';
import { PipelineLike } from '@azure/storage-blob';
import { PipelineOptions } from '@azure/storage-blob';
import type { ProxySettings } from '@azure/core-rest-pipeline';
import type { Readable } from 'stream';
import { RequestPolicy } from '@azure/core-http-compat';
import { RequestPolicyFactory } from '@azure/core-http-compat';
import { RequestPolicyOptionsLike as RequestPolicyOptions } from '@azure/core-http-compat';
import { RestError } from '@azure/core-rest-pipeline';
import { ServiceClientOptions } from '@azure/storage-blob';
import type { ServiceGetPropertiesOptions } from '@azure/storage-blob';
import { ServiceListContainersSegmentResponse } from '@azure/storage-blob';
import type { ServiceRenameContainerOptions } from '@azure/storage-blob';
import type { ServiceSetPropertiesOptions } from '@azure/storage-blob';
import type { ServiceSetPropertiesResponse } from '@azure/storage-blob';
import { StorageBrowserPolicy } from '@azure/storage-blob';
import { StorageBrowserPolicyFactory } from '@azure/storage-blob';
import { StorageRetryOptions } from '@azure/storage-blob';
import { StorageRetryPolicy } from '@azure/storage-blob';
import { StorageRetryPolicyFactory } from '@azure/storage-blob';
import { StorageRetryPolicyType } from '@azure/storage-blob';
import { StorageSharedKeyCredential } from '@azure/storage-blob';
import { StorageSharedKeyCredentialPolicy } from '@azure/storage-blob';
import type { TokenCredential } from '@azure/core-auth';
import type { TransferProgressEvent } from '@azure/core-rest-pipeline';
import type { UserAgentPolicyOptions } from '@azure/core-rest-pipeline';
import { UserDelegationKeyModel } from '@azure/storage-blob';
import { WebResourceLike as WebResource } from '@azure/core-http-compat';
import type { WithResponse } from '@azure/storage-blob';

/**
 * AccessControlChangeCounters contains counts of operations that change Access Control Lists recursively.
 */
export declare interface AccessControlChangeCounters {
    /**
     * Returns number of directories where Access Control List has been updated successfully.
     */
    changedDirectoriesCount: number;
    /**
     * Returns number of files where Access Control List has been updated successfully.
     */
    changedFilesCount: number;
    /**
     * Returns number of paths where Access Control List update has failed.
     */
    failedChangesCount: number;
}

/**
 * Represents an entry that failed to update Access Control List during `setAccessControlRecursive`, `updateAccessControlRecursive` and `removeAccessControlRecursive`.
 */
export declare interface AccessControlChangeError {
    /**
     * Returns name of an entry.
     */
    name: string;
    /**
     * Returns whether entry is a directory.
     */
    isDirectory: boolean;
    /**
     * Returns error message that is the reason why entry failed to update.
     */
    message: string;
}

/**
 * AccessControlChanges contains batch and cumulative counts of operations that change Access Control Lists recursively.
 * Additionally it exposes path entries that failed to update while these operations progress.
 */
export declare interface AccessControlChanges {
    /**
     * Path entries that failed to update Access Control List within single batch.
     */
    batchFailures: AccessControlChangeError[];
    /**
     * Counts of paths changed within single batch.
     */
    batchCounters: AccessControlChangeCounters;
    /**
     * Counts of paths changed from start of the operation.
     */
    aggregateCounters: AccessControlChangeCounters;
    /**
     * Optional. Value is present when operation is split into multiple batches and can be used to resume progress.
     */
    continuationToken?: string;
}

export declare type AccessControlType = "user" | "group" | "mask" | "other";

export declare interface AccessPolicy {
    startsOn?: Date;
    expiresOn?: Date;
    permissions: string;
}

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * This is a helper class to construct a string representing the permissions granted by an AccountSAS. Setting a value
 * to true means that any SAS which uses these permissions will grant permissions for that operation. Once all the
 * values are set, this should be serialized with toString and set as the permissions field on an
 * {@link AccountSASSignatureValues} object. It is possible to construct the permissions string without this class, but
 * the order of the permissions is particular and this class guarantees correctness.
 */
export declare class AccountSASPermissions {
    /**
     * Parse initializes the AccountSASPermissions fields from a string.
     *
     * @param permissions -
     */
    static parse(permissions: string): AccountSASPermissions;
    /**
     * Permission to read resources and list queues and tables granted.
     */
    read: boolean;
    /**
     * Permission to write resources granted.
     */
    write: boolean;
    /**
     * Permission to delete blobs and files granted.
     */
    delete: boolean;
    /**
     * Permission to list blob containers, blobs, shares, directories, and files granted.
     */
    list: boolean;
    /**
     * Permission to add messages, table entities, and append to blobs granted.
     */
    add: boolean;
    /**
     * Permission to create blobs and files granted.
     */
    create: boolean;
    /**
     * Permissions to update messages and table entities granted.
     */
    update: boolean;
    /**
     * Permission to get and delete messages granted.
     */
    process: boolean;
    /**
     * Produces the SAS permissions string for an Azure Storage account.
     * Call this method to set AccountSASSignatureValues Permissions field.
     *
     * Using this method will guarantee the resource types are in
     * an order accepted by the service.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-an-account-sas
     *
     */
    toString(): string;
}

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * This is a helper class to construct a string representing the resources accessible by an AccountSAS. Setting a value
 * to true means that any SAS which uses these permissions will grant access to that resource type. Once all the
 * values are set, this should be serialized with toString and set as the resources field on an
 * {@link AccountSASSignatureValues} object. It is possible to construct the resources string without this class, but
 * the order of the resources is particular and this class guarantees correctness.
 */
export declare class AccountSASResourceTypes {
    /**
     * Creates an {@link AccountSASResourceTypes} from the specified resource types string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid resource type.
     *
     * @param resourceTypes -
     */
    static parse(resourceTypes: string): AccountSASResourceTypes;
    /**
     * Permission to access service level APIs granted.
     */
    service: boolean;
    /**
     * Permission to access container level APIs (Blob Containers, Tables, Queues, File Shares, File Systems) granted.
     */
    container: boolean;
    /**
     * Permission to access object level APIs (Blobs, Table Entities, Queue Messages, Files, Directories) granted.
     */
    object: boolean;
    /**
     * Converts the given resource types to a string.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-an-account-sas
     *
     */
    toString(): string;
}

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * This is a helper class to construct a string representing the services accessible by an AccountSAS. Setting a value
 * to true means that any SAS which uses these permissions will grant access to that service. Once all the
 * values are set, this should be serialized with toString and set as the services field on an
 * {@link AccountSASSignatureValues} object. It is possible to construct the services string without this class, but
 * the order of the services is particular and this class guarantees correctness.
 */
export declare class AccountSASServices {
    /**
     * Creates an {@link AccountSASServices} from the specified services string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid service.
     *
     * @param services -
     */
    static parse(services: string): AccountSASServices;
    /**
     * Permission to access blob and data lake resources granted.
     */
    blob: boolean;
    /**
     * Permission to access file resources granted.
     */
    file: boolean;
    /**
     * Permission to access queue resources granted.
     */
    queue: boolean;
    /**
     * Permission to access table resources granted.
     */
    table: boolean;
    /**
     * Converts the given services to a string.
     *
     */
    toString(): string;
}

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * AccountSASSignatureValues is used to generate a Shared Access Signature (SAS) for an Azure Storage account. Once
 * all the values here are set appropriately, call {@link generateAccountSASQueryParameters} to obtain a representation
 * of the SAS which can actually be applied to data lake urls. Note: that both this class and {@link SASQueryParameters}
 * exist because the former is mutable and a logical representation while the latter is immutable and used to generate
 * actual REST requests.
 *
 * @see https://learn.microsoft.com/en-us/azure/storage/common/storage-dotnet-shared-access-signature-part-1
 * for more conceptual information on SAS
 *
 * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-an-account-sas
 * for descriptions of the parameters, including which are required
 */
export declare interface AccountSASSignatureValues {
    /**
     * If not provided, this defaults to the service version targeted by this version of the library.
     */
    version?: string;
    /**
     * Optional. SAS protocols allowed.
     */
    protocol?: SASProtocol;
    /**
     * Optional. When the SAS will take effect.
     */
    startsOn?: Date;
    /**
     * The time after which the SAS will no longer work.
     */
    expiresOn: Date;
    /**
     * Specifies which operations the SAS user may perform. Please refer to {@link AccountSASPermissions} for help
     * constructing the permissions string.
     */
    permissions: AccountSASPermissions;
    /**
     * Optional. IP range allowed.
     */
    ipRange?: SasIPRange;
    /**
     * Optional. Encryption scope to use when sending requests authorized with this SAS URI.
     */
    encryptionScope?: string;
    /**
     * The values that indicate the services accessible with this SAS. Please refer to {@link AccountSASServices} to
     * construct this value.
     */
    services: string;
    /**
     * The values that indicate the resource types accessible with this SAS. Please refer
     * to {@link AccountSASResourceTypes} to construct this value.
     */
    resourceTypes: string;
}

declare interface AclFailedEntry {
    name?: string;
    type?: string;
    errorMessage?: string;
}

export { AnonymousCredential }

export { AnonymousCredentialPolicy }

export { BaseRequestPolicy }

export declare interface BlobHierarchyListSegment {
    blobPrefixes?: BlobPrefix[];
    blobItems: BlobItemModel[];
}

/** An Azure Storage blob */
export declare interface BlobItemModel {
    name: string;
    deleted: boolean;
    snapshot: string;
    versionId?: string;
    isCurrentVersion?: boolean;
    /** Properties of a blob */
    properties: BlobPropertiesModel;
    deletionId?: string;
}

export declare interface BlobPrefix {
    name: string;
}

/** Properties of a blob */
export declare interface BlobPropertiesModel {
    creationTime?: Date;
    lastModified: Date;
    etag: string;
    /** Size in bytes */
    contentLength?: number;
    contentType?: string;
    contentEncoding?: string;
    contentLanguage?: string;
    contentMD5?: Uint8Array;
    contentDisposition?: string;
    cacheControl?: string;
    blobSequenceNumber?: number;
    copyId?: string;
    copySource?: string;
    copyProgress?: string;
    copyCompletionTime?: Date;
    copyStatusDescription?: string;
    serverEncrypted?: boolean;
    incrementalCopy?: boolean;
    destinationSnapshot?: string;
    deletedTime?: Date;
    remainingRetentionDays?: number;
    accessTierInferred?: boolean;
    customerProvidedKeySha256?: string;
    /** The name of the encryption scope under which the blob is encrypted. */
    encryptionScope?: string;
    accessTierChangeTime?: Date;
    tagCount?: number;
    expiresOn?: Date;
    sealed?: boolean;
    lastAccessedOn?: Date;
}

/**
 * Common options of the {@link FileSystemGenerateSasUrlOptions}, {@link DirectoryGenerateSasUrlOptions}
 * and {@link FileGenerateSasUrlOptions}.
 */
export declare interface CommonGenerateSasUrlOptions {
    /**
     * The version of the service this SAS will target. If not specified, it will default to the version targeted by the
     * library.
     */
    version?: string;
    /**
     * Optional. SAS protocols, HTTPS only or HTTPSandHTTP
     */
    protocol?: SASProtocol;
    /**
     * Optional. When the SAS will take effect.
     */
    startsOn?: Date;
    /**
     * Optional only when identifier is provided. The time after which the SAS will no longer work.
     */
    expiresOn?: Date;
    /**
     * Optional. IP ranges allowed in this SAS.
     */
    ipRange?: SasIPRange;
    /**
     * Optional. The name of the access policy on the container this SAS references if any.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/establishing-a-stored-access-policy
     */
    identifier?: string;
    /**
     * Optional. Encryption scope to use when sending requests authorized with this SAS URI.
     */
    encryptionScope?: string;
    /**
     * Optional. The cache-control header for the SAS.
     */
    cacheControl?: string;
    /**
     * Optional. The content-disposition header for the SAS.
     */
    contentDisposition?: string;
    /**
     * Optional. The content-encoding header for the SAS.
     */
    contentEncoding?: string;
    /**
     * Optional. The content-language header for the SAS.
     */
    contentLanguage?: string;
    /**
     * Optional. The content-type header for the SAS.
     */
    contentType?: string;
}

/**
 * An interface for options common to every remote operation.
 */
export declare interface CommonOptions {
    tracingOptions?: OperationTracingOptions;
}

export declare type CopyStatusType = "pending" | "success" | "aborted" | "failed";

/** Parameter group */
export declare interface CpkInfo {
    /** Optional. Specifies the encryption key to use to encrypt the data provided in the request. If not specified, encryption is performed with the root account encryption key.  For more information, see Encryption at Rest for Azure Storage Services. */
    encryptionKey?: string;
    /** The SHA-256 hash of the provided encryption key. Must be provided if the x-ms-encryption-key header is provided. */
    encryptionKeySha256?: string;
    /** The algorithm used to produce the encryption key hash. Currently, the only accepted value is "AES256". Must be provided if the x-ms-encryption-key header is provided. */
    encryptionAlgorithm?: EncryptionAlgorithmType;
}

export { Credential_2 as Credential }

export { CredentialPolicy }

/**
 * An error thrown when an operation is interrupted and can be continued later on.
 */
export declare class DataLakeAclChangeFailedError extends Error {
    /**
     * Continuation token to continue next batch of operations.
     */
    continuationToken?: string;
    /**
     * Internal error.
     */
    innerError: RestError | Error;
    constructor(error: RestError | Error, continuationToken?: string);
}

/**
 * A DataLakeDirectoryClient represents a URL to the Azure Storage directory.
 */
export declare class DataLakeDirectoryClient extends DataLakePathClient {
    /**
     * Create a directory.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param resourceType - Resource type, must be "directory" for DataLakeDirectoryClient.
     * @param options - Optional. Options when creating directory.
     */
    create(resourceType: PathResourceTypeModel, options?: PathCreateOptions): Promise<PathCreateResponse>;
    /**
     * Create a directory.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param options - Optional. Options when creating directory.
     */
    create(options?: DirectoryCreateOptions): Promise<DirectoryCreateResponse>;
    /**
     * Create a directory if it doesn't already exists.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param resourceType - Resource type, must be "directory" for DataLakeDirectoryClient.
     * @param options -
     */
    createIfNotExists(resourceType: PathResourceTypeModel, options?: PathCreateIfNotExistsOptions): Promise<PathCreateIfNotExistsResponse>;
    /**
     * Create a directory if it doesn't already exists.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param options -
     */
    createIfNotExists(options?: DirectoryCreateIfNotExistsOptions): Promise<DirectoryCreateIfNotExistsResponse>;
    /**
     * Creates a {@link DataLakeDirectoryClient} object under current directory.
     *
     * @param subdirectoryName - Subdirectory name.
     */
    getSubdirectoryClient(subdirectoryName: string): DataLakeDirectoryClient;
    /**
     * Creates a {@link DataLakeFileClient} object under current directory.
     *
     * @param fileName -
     */
    getFileClient(fileName: string): DataLakeFileClient;
    /**
     * Only available for clients constructed with a shared key credential.
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasUrl(options: DirectoryGenerateSasUrlOptions): Promise<string>;
    /**
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasStringToSign(options: DirectoryGenerateSasUrlOptions): string;
    /**
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the input user delegation key.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - Return value of `blobServiceClient.getUserDelegationKey()`
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationSasUrl(options: DirectoryGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): Promise<string>;
    /**
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in The SAS is signed by the input user delegation key.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - Return value of `blobServiceClient.getUserDelegationKey()`
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationSasStringToSign(options: DirectoryGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): string;
}

/**
 * A DataLakeFileClient represents a URL to the Azure Storage file.
 */
export declare class DataLakeFileClient extends DataLakePathClient {
    /**
     * pathContextInternal provided by protocol layer.
     */
    private pathContextInternal;
    /**
     * pathContextInternal provided by protocol layer, with its url pointing to the Blob endpoint.
     */
    private pathContextInternalToBlobEndpoint;
    /**
     * blockBlobClientInternal provided by `@azure/storage-blob` package.
     */
    private blockBlobClientInternal;
    /**
     * Creates an instance of DataLakeFileClient from url and credential.
     *
     * @param url - A Client string pointing to Azure Storage data lake file, such as
     *                     "https://myaccount.dfs.core.windows.net/filesystem/file".
     *                     You can append a SAS if using AnonymousCredential, such as "https://myaccount.dfs.core.windows.net/filesystem/directory/file?sasString".
     * @param credential - Such as AnonymousCredential, StorageSharedKeyCredential or any credential from the `@azure/identity` package to authenticate requests to the service. You can also provide an object that implements the TokenCredential interface. If not specified, AnonymousCredential is used.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(url: string, credential?: StorageSharedKeyCredential | AnonymousCredential | TokenCredential, options?: StoragePipelineOptions);
    /**
     * Creates an instance of DataLakeFileClient from url and pipeline.
     *
     * @param url - A Client string pointing to Azure Storage data lake file, such as
     *                     "https://myaccount.dfs.core.windows.net/filesystem/file".
     *                     You can append a SAS if using AnonymousCredential, such as "https://myaccount.dfs.core.windows.net/filesystem/directory/file?sasString".
     * @param pipeline - Call newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     */
    constructor(url: string, pipeline: Pipeline);
    /**
     * Create a file.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param resourceType - Resource type, must be "file" for DataLakeFileClient.
     * @param options - Optional. Options when creating file.
     */
    create(resourceType: PathResourceTypeModel, options?: PathCreateOptions): Promise<PathCreateResponse>;
    /**
     * Create a file.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param options - Optional. Options when creating file.
     */
    create(options?: FileCreateOptions): Promise<FileCreateResponse>;
    /**
     * Create a file if it doesn't already exists.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param resourceType - Resource type, must be "file" for DataLakeFileClient.
     * @param options -
     */
    createIfNotExists(resourceType: PathResourceTypeModel, options?: PathCreateIfNotExistsOptions): Promise<PathCreateIfNotExistsResponse>;
    /**
     * Create a file if it doesn't already exists.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param options - Optional. Options when creating file.
     */
    createIfNotExists(options?: FileCreateIfNotExistsOptions): Promise<FileCreateIfNotExistsResponse>;
    /**
     * Downloads a file from the service, including its metadata and properties.
     *
     * * In Node.js, data returns in a Readable stream readableStreamBody
     * * In browsers, data returns in a promise contentAsBlob
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/get-blob
     *
     * * Example usage (Node.js):
     *
     * ```js
     * // Download and convert a file to a string
     * const downloadResponse = await fileClient.read();
     * const downloaded = await streamToBuffer(downloadResponse.readableStreamBody);
     * console.log("Downloaded file content:", downloaded.toString());
     *
     * async function streamToBuffer(readableStream) {
     *   return new Promise((resolve, reject) => {
     *     const chunks = [];
     *     readableStream.on("data", (data) => {
     *       chunks.push(typeof data === "string" ? Buffer.from(data) : data);
     *     });
     *     readableStream.on("end", () => {
     *       resolve(Buffer.concat(chunks));
     *     });
     *     readableStream.on("error", reject);
     *   });
     * }
     * ```
     *
     * Example usage (browser):
     *
     * ```js
     * // Download and convert a file to a string
     * const downloadResponse = await fileClient.read();
     * const downloaded = await blobToString(await downloadResponse.contentAsBlob);
     * console.log("Downloaded file content", downloaded);
     *
     * async function blobToString(blob: Blob): Promise<string> {
     *   const fileReader = new FileReader();
     *   return new Promise<string>((resolve, reject) => {
     *     fileReader.onloadend = (ev: any) => {
     *       resolve(ev.target!.result);
     *     };
     *     fileReader.onerror = reject;
     *     fileReader.readAsText(blob);
     *   });
     * }
     * ```
     *
     * @param offset - Optional. Offset to read file, default value is 0.
     * @param count - Optional. How many bytes to read, default will read from offset to the end.
     * @param options - Optional. Options when reading file.
     */
    read(offset?: number, count?: number, options?: FileReadOptions): Promise<FileReadResponse>;
    /**
     * Uploads data to be appended to a file. Data can only be appended to a file.
     * To apply perviously uploaded data to a file, call flush.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/update
     *
     * @param body - Content to be uploaded.
     * @param offset - Append offset in bytes.
     * @param length - Length of content to append in bytes.
     * @param options - Optional. Options when appending data.
     */
    append(body: HttpRequestBody, offset: number, length: number, options?: FileAppendOptions): Promise<FileAppendResponse>;
    /**
     * Flushes (writes) previously appended data to a file.
     *
     * @param position - File position to flush.
     *                          This parameter allows the caller to upload data in parallel and control the order in which it is appended to the file.
     *                          It is required when uploading data to be appended to the file and when flushing previously uploaded data to the file.
     *                          The value must be the position where the data is to be appended. Uploaded data is not immediately flushed, or written,
     *                          to the file. To flush, the previously uploaded data must be contiguous, the position parameter must be specified and
     *                          equal to the length of the file after all data has been written, and there must not be a request entity body included
     *                          with the request.
     * @param options - Optional. Options when flushing data.
     */
    flush(position: number, options?: FileFlushOptions): Promise<FileFlushResponse>;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Uploads a local file to a Data Lake file.
     *
     * @param filePath - Full path of the local file
     * @param options -
     */
    uploadFile(filePath: string, options?: FileParallelUploadOptions): Promise<FileUploadResponse>;
    /**
     * Uploads a Buffer(Node.js)/Blob/ArrayBuffer/ArrayBufferView to a File.
     *
     * @param data - Buffer(Node), Blob, ArrayBuffer or ArrayBufferView
     * @param options -
     */
    upload(data: Buffer | Blob | ArrayBuffer | ArrayBufferView, options?: FileParallelUploadOptions): Promise<FileUploadResponse>;
    private uploadSeekableInternal;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Uploads a Node.js Readable stream into a Data Lake file.
     * This method will try to create a file, then starts uploading chunk by chunk.
     * Please make sure potential size of stream doesn't exceed FILE_MAX_SIZE_BYTES and
     * potential number of chunks doesn't exceed BLOCK_BLOB_MAX_BLOCKS.
     *
     * PERFORMANCE IMPROVEMENT TIPS:
     * * Input stream highWaterMark is better to set a same value with options.chunkSize
     *   parameter, which will avoid Buffer.concat() operations.
     *
     * @param stream - Node.js Readable stream.
     * @param options -
     */
    uploadStream(stream: Readable, options?: FileParallelUploadOptions): Promise<FileUploadResponse>;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Reads a Data Lake file in parallel to a buffer.
     * Offset and count are optional, pass 0 for both to read the entire file.
     *
     * Warning: Buffers can only support files up to about one gigabyte on 32-bit systems or about two
     * gigabytes on 64-bit systems due to limitations of Node.js/V8. For files larger than this size,
     * consider {@link readToFile}.
     *
     * @param buffer - Buffer to be fill, must have length larger than count
     * @param offset - From which position of the Data Lake file to read
     * @param count - How much data to be read. Will read to the end when passing undefined
     * @param options -
     */
    readToBuffer(buffer: Buffer, offset?: number, count?: number, options?: FileReadToBufferOptions): Promise<Buffer>;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME
     *
     * Reads a Data Lake file in parallel to a buffer.
     * Offset and count are optional, pass 0 for both to read the entire file
     *
     * Warning: Buffers can only support files up to about one gigabyte on 32-bit systems or about two
     * gigabytes on 64-bit systems due to limitations of Node.js/V8. For files larger than this size,
     * consider {@link readToFile}.
     *
     * @param offset - From which position of the Data Lake file to read(in bytes)
     * @param count - How much data(in bytes) to be read. Will read to the end when passing undefined
     * @param options -
     */
    readToBuffer(offset?: number, count?: number, options?: FileReadToBufferOptions): Promise<Buffer>;
    /**
     * ONLY AVAILABLE IN NODE.JS RUNTIME.
     *
     * Downloads a Data Lake file to a local file.
     * Fails if the the given file path already exits.
     * Offset and count are optional, pass 0 and undefined respectively to download the entire file.
     *
     * @param filePath -
     * @param offset - From which position of the file to download.
     * @param count - How much data to be downloaded. Will download to the end when passing undefined.
     * @param options - Options to read Data Lake file.
     * @returns The response data for file read operation,
     *                                      but with readableStreamBody set to undefined since its
     *                                      content is already read and written into a local file
     *                                      at the specified path.
     */
    readToFile(filePath: string, offset?: number, count?: number, options?: FileReadOptions): Promise<FileReadResponse>;
    /**
     * Quick query for a JSON or CSV formatted file.
     *
     * Example usage (Node.js):
     *
     * ```js
     * // Query and convert a file to a string
     * const queryResponse = await fileClient.query("select * from BlobStorage");
     * const downloaded = (await streamToBuffer(queryResponse.readableStreamBody)).toString();
     * console.log("Query file content:", downloaded);
     *
     * async function streamToBuffer(readableStream) {
     *   return new Promise((resolve, reject) => {
     *     const chunks = [];
     *     readableStream.on("data", (data) => {
     *       chunks.push(typeof data === "string" ? Buffer.from(data) : data);
     *     });
     *     readableStream.on("end", () => {
     *       resolve(Buffer.concat(chunks));
     *     });
     *     readableStream.on("error", reject);
     *   });
     * }
     * ```
     *
     * @param query -
     * @param options -
     */
    query(query: string, options?: FileQueryOptions): Promise<FileReadResponse>;
    /**
     * Sets an expiry time on a file, once that time is met the file is deleted.
     *
     * @param mode -
     * @param options -
     */
    setExpiry(mode: FileExpiryMode, options?: FileSetExpiryOptions): Promise<FileSetExpiryResponse>;
    /**
     * Only available for clients constructed with a shared key credential.
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasUrl(options: FileGenerateSasUrlOptions): Promise<string>;
    /**
     * Only available for clients constructed with a shared key credential.
     *
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasStringToSign(options: FileGenerateSasUrlOptions): string;
    /**
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the input user delegation key.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - Return value of `blobServiceClient.getUserDelegationKey()`
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationSasUrl(options: FileGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): Promise<string>;
    /**
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the input user delegation key.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - Return value of `blobServiceClient.getUserDelegationKey()`
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationSasStringToSign(options: FileGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): string;
}

/**
 * A DataLakeFileSystemClient represents a URL to the Azure Storage file system
 * allowing you to manipulate its directories and files.
 */
export declare class DataLakeFileSystemClient extends StorageClient {
    /**
     * fileSystemContext provided by protocol layer.
     */
    private fileSystemContext;
    /**
     * fileSystemContext provided by protocol layer.
     */
    private fileSystemContextToBlobEndpoint;
    /**
     * blobContainerClient provided by `@azure/storage-blob` package.
     */
    private blobContainerClient;
    /**
     * Creates an instance of DataLakeFileSystemClient from url and credential.
     *
     * @param url - A Client string pointing to Azure Storage data lake file system, such as
     *                     "https://myaccount.dfs.core.windows.net/filesystem". You can append a SAS
     *                     if using AnonymousCredential, such as "https://myaccount.dfs.core.windows.net/filesystem?sasString".
     * @param credential - Such as AnonymousCredential, StorageSharedKeyCredential or any credential from the `@azure/identity` package to authenticate requests to the service. You can also provide an object that implements the TokenCredential interface. If not specified, AnonymousCredential is used.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(url: string, credential?: StorageSharedKeyCredential | AnonymousCredential | TokenCredential, options?: StoragePipelineOptions);
    /**
     * Creates an instance of DataLakeFileSystemClient from url and pipeline.
     *
     * @param url - A Client string pointing to Azure Storage data lake file system, such as
     *                     "https://myaccount.dfs.core.windows.net/filesystem". You can append a SAS
     *                     if using AnonymousCredential, such as "https://myaccount.dfs.core.windows.net/filesystem?sasString".
     * @param pipeline - Call newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     */
    constructor(url: string, pipeline: Pipeline);
    /**
     * Name of current file system.
     *
     * @readonly
     */
    get name(): string;
    /**
     * Creates a {@link DataLakeDirectoryClient} object under current file system.
     *
     * @param directoryName -
     */
    getDirectoryClient(directoryName: string): DataLakeDirectoryClient;
    /**
     * Creates a {@link DataLakeFileClient} object under current file system.
     *
     * @param fileName -
     */
    getFileClient(fileName: string): DataLakeFileClient;
    /**
     * Get a {@link DataLakeLeaseClient} that manages leases on the file system.
     *
     * @param proposeLeaseId - Optional. Initial proposed lease Id.
     */
    getDataLakeLeaseClient(proposeLeaseId?: string): DataLakeLeaseClient;
    /**
     * Creates a new file system under the specified account. If the file system with
     * the same name already exists, the operation fails.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/create-container
     *
     * @param options - Optional. Options when creating file system.
     */
    create(options?: FileSystemCreateOptions): Promise<FileSystemCreateResponse>;
    /**
     * Creates a new file system under the specified account. If the file system with
     * the same name already exists, it is not changed.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/create-container
     *
     * @param options -
     */
    createIfNotExists(options?: FileSystemCreateOptions): Promise<FileSystemCreateIfNotExistsResponse>;
    /**
     * Returns true if the File system represented by this client exists; false otherwise.
     *
     * NOTE: use this function with care since an existing file system might be deleted by other clients or
     * applications. Vice versa new file system with the same name might be added by other clients or
     * applications after this function completes.
     *
     * @param options -
     */
    exists(options?: FileSystemExistsOptions): Promise<boolean>;
    /**
     * Delete current file system.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/delete-container
     *
     * @param options - Optional. Options when deleting file system.
     */
    delete(options?: FileSystemDeleteOptions): Promise<FileSystemDeleteResponse>;
    /**
     * Delete current file system if it exists.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/delete-container
     *
     * @param options -
     */
    deleteIfExists(options?: FileSystemDeleteOptions): Promise<FileSystemDeleteIfExistsResponse>;
    /**
     * Returns all user-defined metadata and system properties for the specified
     * file system.
     *
     * WARNING: The `metadata` object returned in the response will have its keys in lowercase, even if
     * they originally contained uppercase characters. This differs from the metadata keys returned by
     * the `listFileSystems` method of {@link DataLakeServiceClient} using the `includeMetadata` option, which
     * will retain their original casing.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/get-container-properties
     *
     * @param options - Optional. Options when getting file system properties.
     */
    getProperties(options?: FileSystemGetPropertiesOptions): Promise<FileSystemGetPropertiesResponse>;
    /**
     * Sets one or more user-defined name-value pairs for the specified file system.
     *
     * If no option provided, or no metadata defined in the parameter, the file system
     * metadata will be removed.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/set-container-metadata
     *
     * @param metadata - Replace existing metadata with this value.
     *                              If no value provided the existing metadata will be removed.
     * @param options - Optional. Options when setting file system metadata.
     */
    setMetadata(metadata?: Metadata, options?: FileSystemSetMetadataOptions): Promise<FileSystemSetMetadataResponse>;
    /**
     * Gets the permissions for the specified file system. The permissions indicate
     * whether file system data may be accessed publicly.
     *
     * WARNING: JavaScript Date will potentially lose precision when parsing startsOn and expiresOn strings.
     * For example, new Date("2018-12-31T03:44:23.8827891Z").toISOString() will get "2018-12-31T03:44:23.882Z".
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/get-container-acl
     *
     * @param options - Optional. Options when getting file system access policy.
     */
    getAccessPolicy(options?: FileSystemGetAccessPolicyOptions): Promise<FileSystemGetAccessPolicyResponse>;
    /**
     * Sets the permissions for the specified file system. The permissions indicate
     * whether directories or files in a file system may be accessed publicly.
     *
     * When you set permissions for a file system, the existing permissions are replaced.
     * If no access or containerAcl provided, the existing file system ACL will be
     * removed.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/set-container-acl
     *
     * @param access - Optional. The level of public access to data in the file system.
     * @param fileSystemAcl - Optional. Array of elements each having a unique Id and details of the access policy.
     * @param options - Optional. Options when setting file system access policy.
     */
    setAccessPolicy(access?: PublicAccessType, fileSystemAcl?: SignedIdentifier<AccessPolicy>[], options?: FileSystemSetAccessPolicyOptions): Promise<FileSystemSetAccessPolicyResponse>;
    /**
     * Returns an async iterable iterator to list all the paths (directories and files)
     * under the specified file system.
     *
     * .byPage() returns an async iterable iterator to list the paths in pages.
     *
     * Example using `for await` syntax:
     *
     * ```js
     * // Get the fileSystemClient before you run these snippets,
     * // Can be obtained from `serviceClient.getFileSystemClient("<your-filesystem-name>");`
     * let i = 1;
     * for await (const path of fileSystemClient.listPaths()) {
     *   console.log(`Path ${i++}: ${path.name}, isDirectory?: ${path.isDirectory}`);
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```js
     * let i = 1;
     * let iter = fileSystemClient.listPaths();
     * let pathItem = await iter.next();
     * while (!pathItem.done) {
     *   console.log(`Path ${i++}: ${pathItem.value.name}, isDirectory?: ${pathItem.value.isDirectory}`);
     *   pathItem = await iter.next();
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```js
     * // passing optional maxPageSize in the page settings
     * let i = 1;
     * for await (const response of fileSystemClient.listPaths().byPage({ maxPageSize: 20 })) {
     *   for (const path of response.pathItems) {
     *     console.log(`Path ${i++}: ${path.name}, isDirectory?: ${path.isDirectory}`);
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```js
     * let i = 1;
     * let iterator = fileSystemClient.listPaths().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * // Prints 2 path names
     * for (const path of response.pathItems) {
     *   console.log(`Path ${i++}: ${path.name}, isDirectory?: ${path.isDirectory}`);
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     *
     * // Passing next marker as continuationToken
     *
     * iterator = fileSystemClient.listPaths().byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * // Prints 10 path names
     * for (const path of response.pathItems) {
     *   console.log(`Path ${i++}: ${path.name}, isDirectory?: ${path.isDirectory}`);
     * }
     * ```
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/list-blobs
     *
     * @param options - Optional. Options when listing paths.
     */
    listPaths(options?: ListPathsOptions): PagedAsyncIterableIterator<Path, FileSystemListPathsResponse>;
    private listItems;
    private listSegments;
    private listPathsSegment;
    /**
     * Returns an async iterable iterator to list all the paths (directories and files)
     * under the specified file system.
     *
     * .byPage() returns an async iterable iterator to list the paths in pages.
     *
     * Example using `for await` syntax:
     *
     * ```js
     * // Get the fileSystemClient before you run these snippets,
     * // Can be obtained from `serviceClient.getFileSystemClient("<your-filesystem-name>");`
     * let i = 1;
     * for await (const deletePath of fileSystemClient.listDeletedPaths()) {
     *   console.log(`Path ${i++}: ${deletePath.name}`);
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```js
     * let i = 1;
     * let iter = fileSystemClient.listDeletedPaths();
     * let deletedPathItem = await iter.next();
     * while (!deletedPathItem.done) {
     *   console.log(`Path ${i++}: ${deletedPathItem.value.name}`);
     *   pathItem = await iter.next();
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```js
     * // passing optional maxPageSize in the page settings
     * let i = 1;
     * for await (const response of fileSystemClient.listDeletedPaths().byPage({ maxPageSize: 20 })) {
     *   for (const deletePath of response.pathItems) {
     *     console.log(`Path ${i++}: ${deletePath.name}`);
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```js
     * let i = 1;
     * let iterator = fileSystemClient.listDeletedPaths().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * // Prints 2 path names
     * for (const path of response.pathItems) {
     *   console.log(`Path ${i++}: ${path.name}}`);
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     *
     * // Passing next marker as continuationToken
     *
     * iterator = fileSystemClient.listDeletedPaths().byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * // Prints 10 path names
     * for (const deletePath of response.deletedPathItems) {
     *   console.log(`Path ${i++}: ${deletePath.name}`);
     * }
     * ```
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/list-blobs
     *
     * @param options - Optional. Options when listing deleted paths.
     */
    listDeletedPaths(options?: ListDeletedPathsOptions): PagedAsyncIterableIterator<DeletedPath, FileSystemListDeletedPathsResponse>;
    private listDeletedItems;
    private listDeletedSegments;
    private listDeletedPathsSegment;
    /**
     * Restores a soft deleted path.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/undelete-blob
     *
     * @param deletedPath - Required.  The path of the deleted path.
     *
     * @param deletionId - Required.  The deletion ID associated with the soft deleted path.
     *
     */
    undeletePath(deletedPath: string, deletionId: string, options?: FileSystemUndeletePathOption): Promise<FileSystemUndeletePathResponse>;
    /**
     * Only available for DataLakeFileSystemClient constructed with a shared key credential.
     *
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasUrl(options: FileSystemGenerateSasUrlOptions): Promise<string>;
    /**
     * Only available for DataLakeFileSystemClient constructed with a shared key credential.
     *
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasStringToSign(options: FileSystemGenerateSasUrlOptions): string;
    /**
     * Generates a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the input user delegation key.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - Return value of `blobServiceClient.getUserDelegationKey()`
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationSasUrl(options: FileSystemGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): Promise<string>;
    /**
     * Generates string to sign for a Service Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the input user delegation key.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     * @param options - Optional parameters.
     * @param userDelegationKey - Return value of `blobServiceClient.getUserDelegationKey()`
     * @returns The SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateUserDelegationSasStringToSign(options: FileSystemGenerateSasUrlOptions, userDelegationKey: UserDelegationKey): string;
}

export declare class DataLakeLeaseClient {
    private readonly client;
    get leaseId(): string;
    get url(): string;
    constructor(client: BlobLeaseClient);
    acquireLease(duration: number, options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    changeLease(proposedLeaseId: string, options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    releaseLease(options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
    renewLease(options?: LeaseOperationOptions): Promise<Lease>;
    breakLease(breakPeriod: number, options?: LeaseOperationOptions): Promise<LeaseOperationResponse>;
}

/**
 * A DataLakePathClient represents a URL to the Azure Storage path (directory or file).
 */
export declare class DataLakePathClient extends StorageClient {
    /**
     * pathContext provided by protocol layer.
     */
    private pathContext;
    /**
     * blobClient provided by `@azure/storage-blob` package.
     */
    private blobClient;
    private isTokenCredential?;
    /**
     * SetAccessControlRecursiveInternal operation sets the Access Control on a path and sub paths.
     *
     * @param mode - Mode \"set\" sets POSIX access control rights on files and directories,
     *                                                 Mode \"modify\" modifies one or more POSIX access control rights that pre-exist on files and directories,
     *                                                 Mode \"remove\" removes one or more POSIX access control rights that were present earlier on files and directories.
     * @param acl - The POSIX access control list for the file or directory.
     * @param options - Optional. Options
     */
    private setAccessControlRecursiveInternal;
    /**
     * Creates an instance of DataLakePathClient from url and credential.
     *
     * @param url - A Client string pointing to Azure Storage data lake path (directory or file), such as
     *                     "https://myaccount.dfs.core.windows.net/filesystem/directory" or "https://myaccount.dfs.core.windows.net/filesystem/file".
     *                     You can append a SAS if using AnonymousCredential, such as "https://myaccount.dfs.core.windows.net/filesystem/directory?sasString".
     * @param credential - Such as AnonymousCredential, StorageSharedKeyCredential or any credential from the `@azure/identity` package to authenticate requests to the service. You can also provide an object that implements the TokenCredential interface. If not specified, AnonymousCredential is used.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(url: string, credential?: StorageSharedKeyCredential | AnonymousCredential | TokenCredential, options?: StoragePipelineOptions);
    /**
     * Creates an instance of DataLakePathClient from url and pipeline.
     *
     * @param url - A Client string pointing to Azure Storage data lake path (directory or file), such as
     *                     "https://myaccount.dfs.core.windows.net/filesystem/directory" or "https://myaccount.dfs.core.windows.net/filesystem/file".
     *                     You can append a SAS if using AnonymousCredential, such as "https://myaccount.dfs.core.windows.net/filesystem/directory?sasString".
     * @param pipeline - Call newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     */
    constructor(url: string, pipeline: Pipeline);
    /**
     * Name of current file system.
     *
     * @readonly
     */
    get fileSystemName(): string;
    /**
     * Name of current path (directory or file).
     *
     * @readonly
     */
    get name(): string;
    /**
     * Convert current DataLakePathClient to DataLakeDirectoryClient if current path is a directory.
     *
     */
    toDirectoryClient(): DataLakeDirectoryClient;
    /**
     * Convert current DataLakePathClient to DataLakeFileClient if current path is a file.
     *
     */
    toFileClient(): DataLakeFileClient;
    /**
     * Get a {@link DataLakeLeaseClient} that manages leases on the path (directory or file).
     *
     * @param proposeLeaseId - Optional. Initial proposed lease Id.
     */
    getDataLakeLeaseClient(proposeLeaseId?: string): DataLakeLeaseClient;
    /**
     * Create a directory or path.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param resourceType - Resource type, "directory" or "file".
     * @param options - Optional. Options when creating path.
     */
    create(resourceType: PathResourceTypeModel, options?: PathCreateOptions): Promise<PathCreateResponse>;
    /**
     * Create a directory or file. If the resource already exists, it is not changed.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param resourceType - Resource type, "directory" or "file".
     * @param options -
     */
    createIfNotExists(resourceType: PathResourceTypeModel, options?: PathCreateIfNotExistsOptions): Promise<PathCreateIfNotExistsResponse>;
    /**
     * Returns true if the Data Lake file represented by this client exists; false otherwise.
     *
     * NOTE: use this function with care since an existing file might be deleted by other clients or
     * applications. Vice versa new files might be added by other clients or applications after this
     * function completes.
     *
     * @param options - options to Exists operation.
     */
    exists(options?: PathExistsOptions): Promise<boolean>;
    /**
     * Delete current path (directory or file).
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/delete
     *
     * @param recursive - Required and valid only when the resource is a directory. If "true", all paths beneath the directory will be deleted.
     * @param options - Optional. Options when deleting path.
     */
    delete(recursive?: boolean, options?: PathDeleteOptions): Promise<PathDeleteResponse>;
    /**
     * Delete current path (directory or file) if it exists.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/delete
     *
     * @param recursive - Required and valid only when the resource is a directory. If "true", all paths beneath the directory will be deleted.
     * @param options -
     */
    deleteIfExists(recursive?: boolean, options?: PathDeleteOptions): Promise<PathDeleteIfExistsResponse>;
    /**
     * Returns the access control data for a path (directory of file).
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/getproperties
     *
     * @param options - Optional. Options when getting file access control.
     */
    getAccessControl(options?: PathGetAccessControlOptions): Promise<PathGetAccessControlResponse>;
    /**
     * Set the access control data for a path (directory of file).
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/update
     *
     * @param acl - The POSIX access control list for the file or directory.
     * @param options - Optional. Options when setting path access control.
     */
    setAccessControl(acl: PathAccessControlItem[], options?: PathSetAccessControlOptions): Promise<PathSetAccessControlResponse>;
    /**
     * Sets the Access Control on a path and sub paths.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/update
     *
     * @param acl - The POSIX access control list for the file or directory.
     * @param options - Optional. Options
     */
    setAccessControlRecursive(acl: PathAccessControlItem[], options?: PathChangeAccessControlRecursiveOptions): Promise<PathChangeAccessControlRecursiveResponse>;
    /**
     * Modifies the Access Control on a path and sub paths.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/update
     *
     * @param acl - The POSIX access control list for the file or directory.
     * @param options - Optional. Options
     */
    updateAccessControlRecursive(acl: PathAccessControlItem[], options?: PathChangeAccessControlRecursiveOptions): Promise<PathChangeAccessControlRecursiveResponse>;
    /**
     * Removes the Access Control on a path and sub paths.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/update
     *
     * @param acl - The POSIX access control list for the file or directory.
     * @param options - Optional. Options
     */
    removeAccessControlRecursive(acl: RemovePathAccessControlItem[], options?: PathChangeAccessControlRecursiveOptions): Promise<PathChangeAccessControlRecursiveResponse>;
    /**
     * Sets the file permissions on a path.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/update
     *
     * @param permissions - The POSIX access permissions for the file owner, the file owning group, and others.
     * @param options - Optional. Options when setting path permissions.
     */
    setPermissions(permissions: PathPermissions, options?: PathSetPermissionsOptions): Promise<PathSetPermissionsResponse>;
    /**
     * Returns all user-defined metadata, standard HTTP properties, and system properties
     * for the path (directory or file).
     *
     * WARNING: The `metadata` object returned in the response will have its keys in lowercase, even if
     * they originally contained uppercase characters. This differs from the metadata keys returned by
     * the methods of {@link DataLakeFileSystemClient} that list paths using the `includeMetadata` option, which
     * will retain their original casing.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/get-blob-properties
     *
     * @param options - Optional. Options when getting path properties.
     */
    getProperties(options?: PathGetPropertiesOptions): Promise<PathGetPropertiesResponse>;
    /**
     * Sets system properties on the path (directory or file).
     *
     * If no value provided, or no value provided for the specified blob HTTP headers,
     * these blob HTTP headers without a value will be cleared.
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/set-blob-properties
     *
     * @param httpHeaders -
     * @param options -
     */
    setHttpHeaders(httpHeaders: PathHttpHeaders, options?: PathSetHttpHeadersOptions): Promise<PathSetHttpHeadersResponse>;
    /**
     * Sets user-defined metadata for the specified path (directory of file) as one or more name-value pairs.
     *
     * If no option provided, or no metadata defined in the parameter, the path
     * metadata will be removed.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/set-blob-metadata
     *
     * @param metadata - Optional. Replace existing metadata with this value.
     *                              If no value provided the existing metadata will be removed.
     * @param options - Optional. Options when setting path metadata.
     */
    setMetadata(metadata?: Metadata, options?: PathSetMetadataOptions): Promise<PathSetMetadataResponse>;
    /**
     * Move directory or file within same file system.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param destinationPath - Destination directory path like "directory" or file path "directory/file".
     *                                 If the destinationPath is authenticated with SAS, add the SAS to the destination path like "directory/file?sasToken".
     * @param options - Optional. Options when moving directory or file.
     */
    move(destinationPath: string, options?: PathMoveOptions): Promise<PathMoveResponse>;
    /**
     * Move directory or file to another file system.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/datalakestoragegen2/path/create
     *
     * @param destinationFileSystem - Destination file system like "filesystem".
     * @param destinationPath - Destination directory path like "directory" or file path "directory/file"
     *                                 If the destinationPath is authenticated with SAS, add the SAS to the destination path like "directory/file?sasToken".
     * @param options - Optional. Options when moving directory or file.
     */
    move(destinationFileSystem: string, destinationPath: string, options?: PathMoveOptions): Promise<PathMoveResponse>;
}

export declare interface DataLakeRequestConditions extends ModifiedAccessConditions, LeaseAccessConditions {
}

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * This is a helper class to construct a string representing the permissions granted by a ServiceSAS. Setting
 * a value to true means that any SAS which uses these permissions will grant permissions for that operation. Once all
 * the values are set, this should be serialized with toString and set as the permissions field on a
 * {@link DataLakeSASSignatureValues} object. It is possible to construct the permissions string without this class, but
 * the order of the permissions is particular and this class guarantees correctness.
 */
export declare class DataLakeSASPermissions {
    /**
     * Creates a {@link DataLakeSASPermissions} from the specified permissions string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid permission.
     *
     * @param permissions -
     */
    static parse(permissions: string): DataLakeSASPermissions;
    /**
     * Specifies Read access granted.
     */
    read: boolean;
    /**
     * Specifies Add access granted.
     */
    add: boolean;
    /**
     * Specifies Create access granted.
     */
    create: boolean;
    /**
     * Specifies Write access granted.
     */
    write: boolean;
    /**
     * Specifies Delete access granted.
     */
    delete: boolean;
    /**
     * Specifies Move access granted.
     */
    move: boolean;
    /**
     * Specifies Execute access granted.
     */
    execute: boolean;
    /**
     * Specifies Ownership access granted, which allows the caller to set owner, owning group,
     * or act as the owner when renaming or deleting a blob (file or directory) within a folder
     * that has the sticky bit set.
     */
    manageOwnership: boolean;
    /**
     * Specifies Permission access granted, which allows the caller to set permissions and
     * POSIX ACLs on blobs (files and directories).
     */
    manageAccessControl: boolean;
    /**
     * Converts the given permissions to a string. Using this method will guarantee the permissions are in an
     * order accepted by the service.
     *
     * @returns A string which represents the DataLakeSASPermissions
     */
    toString(): string;
}

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * DataLakeSASSignatureValues is used to help generating Blob and DataLake service SAS tokens for containers, blobs, filesystem, directories and files.
 */
export declare interface DataLakeSASSignatureValues {
    /**
     * The version of the service this SAS will target. If not specified, it will default to the version targeted by the
     * library.
     */
    version?: string;
    /**
     * Optional. SAS protocols, HTTPS only or HTTPSandHTTP
     */
    protocol?: SASProtocol;
    /**
     * Optional. When the SAS will take effect.
     */
    startsOn?: Date;
    /**
     * Optional only when identifier is provided. The time after which the SAS will no longer work.
     */
    expiresOn?: Date;
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link FileSystemSASPermissions}, {@link DirectorySASPermissions} or {@link DataLakeSASPermissions} depending on the resource
     * being accessed for help constructing the permissions string.
     */
    permissions?: DataLakeSASPermissions | DirectorySASPermissions | FileSystemSASPermissions;
    /**
     * Optional. IP ranges allowed in this SAS.
     */
    ipRange?: SasIPRange;
    /**
     * The name of the file system the SAS user may access.
     */
    fileSystemName: string;
    /**
     * Optional. The path name of the directory or file SAS user may access. Required if snapshotTime is provided.
     */
    pathName?: string;
    /**
     * Optional. Beginning in version 2020-02-10, this value defines whether or not the {@link pathName} is a directory.
     * If this value is set to true, the Path is a Directory for a Directory SAS. If set to false or default, the Path
     * is a File Path for a File Path SAS.
     */
    isDirectory?: boolean;
    /**
     * Optional. Beginning in version 2020-02-10, indicate the depth of the directory specified in the canonicalizedresource field of the string-to-sign.
     * The depth of the directory is the number of directories beneath the root folder.
     */
    directoryDepth?: number;
    /**
     * Optional. Beginning in version 2020-02-10, specifies the Authorized AAD Object Id in GUID format. The AAD Object ID of a user
     * authorized by the owner of the user delegation key to perform the action granted by the SAS. The Azure Storage service will
     * ensure that the owner of the user delegation key has the required permissions before granting access but no additional permission
     * check for the user specified in this value will be performed. This cannot be used in conjuction with {@link agentObjectId}.
     * This is only used for User Delegation SAS.
     */
    preauthorizedAgentObjectId?: string;
    /**
     * Optional. Beginning in version 2020-02-10, specifies the Unauthorized AAD Object Id in GUID format. The AAD Object Id of a user that is assumed
     * to be unauthorized by the owner of the user delegation key. The Azure Storage Service will perform an additional POSIX ACL check to determine
     * if the user is authorized to perform the requested operation. This cannot be used in conjuction with {@link preauthorizedAgentObjectId}.
     * This is only used for User Delegation SAS.
     */
    agentObjectId?: string;
    /**
     * Optional. Beginning in version 2020-02-10, this is a GUID value that will be logged in the storage diagnostic logs and can be used to
     * correlate SAS generation with storage resource access. This is only used for User Delegation SAS.
     */
    correlationId?: string;
    /**
     * Optional. Snapshot timestamp string the SAS user may access. Only supported from API version 2018-11-09.
     */
    snapshotTime?: string;
    /**
     * Optional. The name of the access policy on the file system this SAS references if any.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/establishing-a-stored-access-policy
     */
    identifier?: string;
    /**
     * Optional. Encryption scope to use when sending requests authorized with this SAS URI.
     */
    encryptionScope?: string;
    /**
     * Optional. The cache-control header for the SAS.
     */
    cacheControl?: string;
    /**
     * Optional. The content-disposition header for the SAS.
     */
    contentDisposition?: string;
    /**
     * Optional. The content-encoding header for the SAS.
     */
    contentEncoding?: string;
    /**
     * Optional. The content-language header for the SAS.
     */
    contentLanguage?: string;
    /**
     * Optional. The content-type header for the SAS.
     */
    contentType?: string;
}

/**
 * DataLakeServiceClient allows you to manipulate Azure
 * Data Lake service resources and file systems. The storage account provides
 * the top-level namespace for the Data Lake service.
 */
export declare class DataLakeServiceClient extends StorageClient {
    /**
     * blobServiceClient provided by `@azure/storage-blob` package.
     */
    private blobServiceClient;
    /**
     *
     * Creates an instance of DataLakeServiceClient from connection string.
     *
     * @param connectionString - Account connection string or a SAS connection string of an Azure storage account.
     *                                  [ Note - Account connection string can only be used in NODE.JS runtime. ]
     *                                  Account connection string example -
     *                                  `DefaultEndpointsProtocol=https;AccountName=myaccount;AccountKey=accountKey;EndpointSuffix=core.windows.net`
     *                                  SAS connection string example -
     *                                  `BlobEndpoint=https://myaccount.blob.core.windows.net/;QueueEndpoint=https://myaccount.queue.core.windows.net/;FileEndpoint=https://myaccount.file.core.windows.net/;TableEndpoint=https://myaccount.table.core.windows.net/;SharedAccessSignature=sasString`
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    static fromConnectionString(connectionString: string, options?: StoragePipelineOptions): DataLakeServiceClient;
    /**
     * Creates an instance of DataLakeServiceClient from url.
     *
     * @param url - A Client string pointing to Azure Storage data lake service, such as
     *                     "https://myaccount.dfs.core.windows.net". You can append a SAS
     *                     if using AnonymousCredential, such as "https://myaccount.dfs.core.windows.net?sasString".
     * @param credential - Such as AnonymousCredential, StorageSharedKeyCredential or any credential from the `@azure/identity` package to authenticate requests to the service. You can also provide an object that implements the TokenCredential interface. If not specified, AnonymousCredential is used.
     * @param options - Optional. Options to configure the HTTP pipeline.
     */
    constructor(url: string, credential?: StorageSharedKeyCredential | AnonymousCredential | TokenCredential, options?: StoragePipelineOptions);
    /**
     * Creates an instance of DataLakeServiceClient from url and pipeline.
     *
     * @param url - A Client string pointing to Azure Storage data lake service, such as
     *                     "https://myaccount.dfs.core.windows.net". You can append a SAS
     *                     if using AnonymousCredential, such as "https://myaccount.dfs.core.windows.net?sasString".
     * @param pipeline - Call newPipeline() to create a default
     *                            pipeline, or provide a customized pipeline.
     */
    constructor(url: string, pipeline: Pipeline);
    /**
     * Creates a {@link DataLakeFileSystemClient} object.
     *
     * @param fileSystemName - File system name.
     */
    getFileSystemClient(fileSystemName: string): DataLakeFileSystemClient;
    /**
     * ONLY AVAILABLE WHEN USING BEARER TOKEN AUTHENTICATION (TokenCredential).
     *
     * Retrieves a user delegation key for the Data Lake service. This is only a valid operation when using
     * bearer token authentication.
     *
     * @example
     * ```js
     * // Generate user delegation SAS for a file system
     * const userDelegationKey = await dataLakeServiceClient.getUserDelegationKey(startsOn, expiresOn);
     * const fileSystemSAS = generateDataLakeSASQueryParameters({
     *     fileSystemName, // Required
     *     permissions: FileSystemSASPermissions.parse("racwdl"), // Required
     *     startsOn, // Required. Date type
     *     expiresOn, // Optional. Date type
     *     ipRange: { start: "0.0.0.0", end: "255.255.255.255" }, // Optional
     *     protocol: SASProtocol.HttpsAndHttp, // Optional
     *     version: "2018-11-09" // Must greater than or equal to 2018-11-09 to generate user delegation SAS
     *   },
     *   userDelegationKey, // UserDelegationKey
     *   accountName
     * ).toString();
     * ```
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/get-user-delegation-key
     *
     * @param startsOn - The start time for the user delegation SAS. Must be within 7 days of the current time.
     * @param expiresOn - The end time for the user delegation SAS. Must be within 7 days of the current time.
     * @param options -
     */
    getUserDelegationKey(startsOn: Date, expiresOn: Date, options?: ServiceGetUserDelegationKeyOptions): Promise<ServiceGetUserDelegationKeyResponse>;
    /**
     * Returns an async iterable iterator to list all the file systems
     * under the specified account.
     *
     * .byPage() returns an async iterable iterator to list the file systems in pages.
     *
     * Example using `for await` syntax:
     *
     * ```js
     * let i = 1;
     * for await (const fileSystem of serviceClient.listFileSystems()) {
     *   console.log(`FileSystem ${i++}: ${fileSystem.name}`);
     * }
     * ```
     *
     * Example using `iter.next()`:
     *
     * ```js
     * let i = 1;
     * const iter = serviceClient.listFileSystems();
     * let fileSystemItem = await iter.next();
     * while (!fileSystemItem.done) {
     *   console.log(`FileSystem ${i++}: ${fileSystemItem.value.name}`);
     *   fileSystemItem = await iter.next();
     * }
     * ```
     *
     * Example using `byPage()`:
     *
     * ```js
     * // passing optional maxPageSize in the page settings
     * let i = 1;
     * for await (const response of serviceClient.listFileSystems().byPage({ maxPageSize: 20 })) {
     *   if (response.fileSystemItems) {
     *     for (const fileSystem of response.fileSystemItems) {
     *       console.log(`FileSystem ${i++}: ${fileSystem.name}`);
     *     }
     *   }
     * }
     * ```
     *
     * Example using paging with a marker:
     *
     * ```js
     * let i = 1;
     * let iterator = serviceClient.listFileSystems().byPage({ maxPageSize: 2 });
     * let response = (await iterator.next()).value;
     *
     * // Prints 2 file system names
     * if (response.fileSystemItems) {
     *   for (const fileSystem of response.fileSystemItems) {
     *     console.log(`FileSystem ${i++}: ${fileSystem.name}`);
     *   }
     * }
     *
     * // Gets next marker
     * let marker = response.continuationToken;
     * // Passing next marker as continuationToken
     * iterator = serviceClient
     *   .listContainers()
     *   .byPage({ continuationToken: marker, maxPageSize: 10 });
     * response = (await iterator.next()).value;
     *
     * // Prints 10 file system names
     * if (response.fileSystemItems) {
     *   for (const fileSystem of response.fileSystemItems) {
     *      console.log(`FileSystem ${i++}: ${fileSystem.name}`);
     *   }
     * }
     * ```
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/list-containers2
     *
     * @param options -
     */
    listFileSystems(options?: ServiceListFileSystemsOptions): PagedAsyncIterableIterator<FileSystemItem, ServiceListFileSystemsSegmentResponse>;
    /**
     * Only available for DataLakeServiceClient constructed with a shared key credential.
     *
     * Generates an account Shared Access Signature (SAS) URI based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/create-account-sas
     *
     * @param expiresOn - Optional. The time at which the shared access signature becomes invalid. Default to an hour later if not specified.
     * @param permissions - Specifies the list of permissions to be associated with the SAS.
     * @param resourceTypes - Specifies the resource types associated with the shared access signature.
     * @param options - Optional parameters.
     * @returns An account SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateAccountSasUrl(expiresOn?: Date, permissions?: AccountSASPermissions, resourceTypes?: string, options?: ServiceGenerateAccountSasUrlOptions): string;
    /**
     * Only available for DataLakeServiceClient constructed with a shared key credential.
     *
     * Generates string to sign for an account Shared Access Signature (SAS) based on the client properties
     * and parameters passed in. The SAS is signed by the shared key credential of the client.
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/create-account-sas
     *
     * @param expiresOn - Optional. The time at which the shared access signature becomes invalid. Default to an hour later if not specified.
     * @param permissions - Specifies the list of permissions to be associated with the SAS.
     * @param resourceTypes - Specifies the resource types associated with the shared access signature.
     * @param options - Optional parameters.
     * @returns An account SAS URI consisting of the URI to the resource represented by this client, followed by the generated SAS token.
     */
    generateSasStringToSign(expiresOn?: Date, permissions?: AccountSASPermissions, resourceTypes?: string, options?: ServiceGenerateAccountSasUrlOptions): string;
    /**
     * Renames an existing File System.
     *
     * @param sourceFileSystemName - The name of the source File System.
     * @param destinationContainerName - The new name of the File System.
     * @param options - Options to configure File System Rename operation.
     */
    private renameFileSystem;
    /**
     * Restore a previously deleted File System.
     * This API is only functional if Container Soft Delete is enabled for the storage account.
     *
     * @param deletedFileSystemName - The name of the source File System.
     * @param deleteFileSystemVersion - The new name of the File System.
     * @param options - Options to configure File System Restore operation.
     */
    undeleteFileSystem(deletedFileSystemName: string, deleteFileSystemVersion: string, options?: ServiceUndeleteFileSystemOptions): Promise<{
        fileSystemClient: DataLakeFileSystemClient;
        fileSystemUndeleteResponse: FileSystemUndeleteResponse;
    }>;
    /**
     * Gets the properties of a storage account’s Blob service endpoint, including properties
     * for Storage Analytics and CORS (Cross-Origin Resource Sharing) rules.
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/get-blob-service-properties
     *
     * @param options - Options to the Service Get Properties operation.
     * @returns Response data for the Service Get Properties operation.
     */
    getProperties(options?: ServiceGetPropertiesOptions): Promise<DataLakeServiceGetPropertiesResponse>;
    /**
     * Sets properties for a storage account’s Blob service endpoint, including properties
     * for Storage Analytics, CORS (Cross-Origin Resource Sharing) rules and soft delete settings.
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/set-blob-service-properties
     *
     * @param properties -
     * @param options - Options to the Service Set Properties operation.
     * @returns Response data for the Service Set Properties operation.
     */
    setProperties(properties: DataLakeServiceProperties, options?: ServiceSetPropertiesOptions): Promise<ServiceSetPropertiesResponse>;
}

export { DataLakeServiceGetPropertiesResponse }

export { DataLakeServiceProperties }

export declare interface DeletedPath {
    name: string;
    deletionId?: string;
    deletedOn?: Date;
    remainingRetentionDays?: number;
}

export declare interface DeletedPathList {
    pathItems?: DeletedPath[];
}

export declare interface DirectoryCreateIfNotExistsOptions extends PathCreateIfNotExistsOptions {
}

export declare interface DirectoryCreateIfNotExistsResponse extends PathCreateIfNotExistsResponse {
}

/** **************************************************************/
/** DataLakeDirectoryClient option and response related models **/
/** **************************************************************/
export declare interface DirectoryCreateOptions extends PathCreateOptions {
}

export declare interface DirectoryCreateResponse extends PathCreateResponse {
}

/**
 * Options to configure {@link DataLakeDirectoryClient.generateSasUrl} operation.
 */
export declare interface DirectoryGenerateSasUrlOptions extends CommonGenerateSasUrlOptions {
    /**
     * Optional only when identifier is provided. Specifies the list of permissions to be associated with the SAS.
     */
    permissions?: DirectorySASPermissions;
}

/**
 * This is a helper class to construct a string representing the permissions granted by a ServiceSAS to a directory.
 * Setting a value to true means that any SAS which uses these permissions will grant permissions for that operation.
 * Once all the values are set, this should be serialized with toString and set as the permissions field on a
 * {@link DataLakeSASSignatureValues} object. It is possible to construct the permissions string without this class, but
 * the order of the permissions is particular and this class guarantees correctness.
 */
export declare class DirectorySASPermissions {
    /**
     * Creates an {@link DirectorySASPermissions} from the specified permissions string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid permission.
     *
     * @param permissions -
     */
    static parse(permissions: string): DirectorySASPermissions;
    /**
     * Specifies Read access granted.
     */
    read: boolean;
    /**
     * Specifies Add access granted.
     */
    add: boolean;
    /**
     * Specifies Create access granted.
     */
    create: boolean;
    /**
     * Specifies Write access granted.
     */
    write: boolean;
    /**
     * Specifies Delete access granted.
     */
    delete: boolean;
    /**
     * Specifies List access granted.
     */
    list: boolean;
    /**
     * Specifies Move access granted.
     */
    move: boolean;
    /**
     * Specifies Execute access granted.
     */
    execute: boolean;
    /**
     * Specifies Ownership access granted, which allows the caller to set owner, owning group,
     * or act as the owner when renaming or deleting a blob (file or directory) within a folder
     * that has the sticky bit set.
     */
    manageOwnership: boolean;
    /**
     * Specifies Permission access granted, which allows the caller to set permissions and
     * POSIX ACLs on blobs (files and directories).
     */
    manageAccessControl: boolean;
    /**
     * Converts the given permissions to a string. Using this method will guarantee the permissions are in an
     * order accepted by the service.
     *
     * The order of the characters should be as specified here to ensure correctness.
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     */
    toString(): string;
}

/**
 * Defines values for EncryptionAlgorithmType. \
 * {@link KnownEncryptionAlgorithmType} can be used interchangeably with EncryptionAlgorithmType,
 *  this enum contains the known values that the service supports.
 * ### Known values supported by the service
 * **None** \
 * **AES256**
 */
export declare type EncryptionAlgorithmType = string;

export declare interface FileAppendOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: LeaseAccessConditions;
    transactionalContentMD5?: Uint8Array;
    onProgress?: (progress: TransferProgressEvent) => void;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
    /**
     * If file should be flushed automatically after the append
     */
    flush?: boolean;
    /**
     * Proposed lease ID, in a GUID string format. The Blob service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats.
     * */
    proposedLeaseId?: string;
    /**
     * The lease duration is required to acquire a lease, and specifies the duration of the lease in seconds.  The lease duration must be between 15 and 60 seconds or -1 for infinite lease.
     * */
    leaseDurationInSeconds?: number;
    /**
     * Optional. If "acquire" it will acquire the lease. If "auto-renew" it will renew the lease. If "release" it will release the lease only on flush. If "acquire-release" it will acquire & complete the operation & release the lease once operation is done.
     * */
    leaseAction?: LeaseAction;
}

export declare type FileAppendResponse = WithResponse<PathAppendDataHeaders, PathAppendDataHeaders>;

export declare interface FileCreateIfNotExistsOptions extends PathCreateIfNotExistsOptions {
}

export declare interface FileCreateIfNotExistsResponse extends PathCreateIfNotExistsResponse {
}

export declare interface FileCreateOptions extends PathCreateOptions {
}

export declare interface FileCreateResponse extends PathCreateResponse {
}

/** Defines values for PathExpiryOptions. */
export declare type FileExpiryMode = "NeverExpire" | "RelativeToCreation" | "RelativeToNow" | "Absolute";

export declare interface FileFlushOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
    retainUncommittedData?: boolean;
    close?: boolean;
    pathHttpHeaders?: PathHttpHeaders;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
    /**
     * Proposed lease ID, in a GUID string format. The Blob service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats.
     */
    proposedLeaseId?: string;
    /**
     * The lease duration is required to acquire a lease, and specifies the duration of the lease in seconds.  The lease duration must be between 15 and 60 seconds or -1 for infinite lease.
     */
    leaseDurationInSeconds?: number;
    /**
     * Optional. If "acquire" it will acquire the lease. If "auto-renew" it will renew the lease. If "release" it will release the lease only on flush. If "acquire-release" it will acquire & complete the operation & release the lease once operation is done.
     */
    leaseAction?: LeaseAction;
}

export declare type FileFlushResponse = WithResponse<PathFlushDataHeaders, PathFlushDataHeaders>;

/**
 * Options to configure {@link DataLakeFileClient.generateSasUrl} operation.
 */
export declare interface FileGenerateSasUrlOptions extends CommonGenerateSasUrlOptions {
    /**
     * Optional only when identifier is provided. Specifies the list of permissions to be associated with the SAS.
     */
    permissions?: DataLakeSASPermissions;
}

/**
 * Option interface for Data Lake file - Upload operations
 *
 * See:
 * - {@link DataLakeFileClient.upload}
 * - {@link DataLakeFileClient.uploadFile}
 * - {@link DataLakeFileClient.uploadStream}
 */
export declare interface FileParallelUploadOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Access conditions headers.
     */
    conditions?: DataLakeRequestConditions;
    /**
     * Http headers.
     */
    pathHttpHeaders?: PathHttpHeaders;
    /**
     * A collection of key-value string pair to associate with the Data Lake file.
     */
    metadata?: Metadata;
    /**
     * Sets POSIX access permissions for the file owner, the file owning group, and others.
     * Each class may be granted read, write, or execute permission. The sticky bit is also supported.
     * Both symbolic (rwxrw-rw-) and 4-digit octal notation (e.g. 0766) are supported.
     */
    permissions?: string;
    /**
     * The umask restricts the permissions of the file to be created.
     * The resulting permission is given by p & ^u, where p is the permission and u is the umask.
     * For example, if p is 0777 and u is 0057, then the resulting permission is 0720.
     * The default permission is 0666 for a file. The default umask is 0027.
     * The umask must be specified in 4-digit octal notation (e.g. 0766).
     */
    umask?: string;
    /**
     * Progress updater.
     */
    onProgress?: (progress: TransferProgressEvent) => void;
    /**
     * When Azure Storage Events are enabled, a file changed event is raised.
     * This event has a property indicating whether this is the final change
     * to distinguish the difference between an intermediate flush to a file stream (when close set to "false")
     * and the final close of a file stream (when close set to "true").
     */
    close?: boolean;
    /**
     * Data size threshold in bytes to use a single upload operation rather than parallel uploading.
     * Data of smaller size than this limit will be transferred in a single upload.
     * Data larger than this limit will be transferred in chunks in parallel.
     * Its default and max value is FILE_MAX_SINGLE_UPLOAD_THRESHOLD.
     * Note: {@link DataLakeFileClient.uploadStream} do not respect this field and always do parallel uploading.
     */
    singleUploadThreshold?: number;
    /**
     * The size of data in bytes that will be transferred in parallel.
     * If set to 0 or undefined, it will be automatically calculated according
     * to the data size. Its max value is FILE_UPLOAD_MAX_CHUNK_SIZE.
     */
    chunkSize?: number;
    /**
     * Max concurrency of parallel uploading. Must be greater than or equal to 0. Its default value is DEFAULT_HIGH_LEVEL_CONCURRENCY.
     */
    maxConcurrency?: number;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
    /**
     * Specifies the encryption context to set on the file.
     */
    encryptionContext?: string;
}

/**
 * Options to query file with Apache Arrow format. Only valid for {@link FileQueryOptions.outputTextConfiguration}.
 */
export declare type FileQueryArrowConfiguration = BlobQueryArrowConfiguration;

/**
 * Options to query file with CSV format.
 */
export declare interface FileQueryCsvTextConfiguration {
    /**
     * Record separator.
     */
    recordSeparator: string;
    /**
     * Query for a CSV format file.
     */
    kind: "csv";
    /**
     * Column separator. Default is ",".
     */
    columnSeparator?: string;
    /**
     * Field quote.
     */
    fieldQuote?: string;
    /**
     * Escape character.
     */
    escapeCharacter?: string;
    /**
     * Has headers. Default is false.
     */
    hasHeaders?: boolean;
}

/**
 * File query error type.
 */
export declare interface FileQueryError {
    /**
     * Whether the error is fatal or not. A fatal error will stop the query.
     */
    isFatal: boolean;
    /**
     * Error name.
     */
    name: string;
    /**
     * Position in bytes of the query.
     */
    position: number;
    /**
     * Error description.
     */
    description: string;
}

/**
 * Options to query file with JSON format.
 */
export declare interface FileQueryJsonTextConfiguration {
    /**
     * Record separator.
     */
    recordSeparator: string;
    /**
     * Query for a JSON format file.
     */
    kind: "json";
}

/**
 * Option interface for Data Lake file - query operations
 *
 * See:
 * - {@link DataLakeFileClient.query}
 */
export declare interface FileQueryOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Configurations for the query input.
     */
    inputTextConfiguration?: FileQueryJsonTextConfiguration | FileQueryCsvTextConfiguration | FileQueryParquetConfiguration;
    /**
     * Configurations for the query output.
     */
    outputTextConfiguration?: FileQueryJsonTextConfiguration | FileQueryCsvTextConfiguration | FileQueryArrowConfiguration;
    /**
     * Callback to receive events on the progress of query operation.
     */
    onProgress?: (progress: TransferProgressEvent) => void;
    /**
     * Callback to receive error events during the query operaiton.
     */
    onError?: (error: FileQueryError) => void;
    /**
     * Conditions to meet when uploading to the block file.
     */
    conditions?: DataLakeRequestConditions;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
}

/**
 * Options to query file with Parquet format.
 */
export declare interface FileQueryParquetConfiguration {
    /**
     * Kind.
     */
    kind: "parquet";
}

export declare interface FileReadHeaders {
    lastModified?: Date;
    /**
     * Returns the date and time the file was created.
     */
    createdOn?: Date;
    metadata?: Metadata;
    contentLength?: number;
    contentType?: string;
    contentRange?: string;
    etag?: string;
    contentMD5?: Uint8Array;
    contentEncoding?: string;
    cacheControl?: string;
    contentDisposition?: string;
    contentLanguage?: string;
    copyCompletedOn?: Date;
    copyStatusDescription?: string;
    copyId?: string;
    copyProgress?: string;
    copySource?: string;
    copyStatus?: CopyStatusType;
    leaseDuration?: LeaseDurationType;
    leaseState?: LeaseStateType;
    leaseStatus?: LeaseStatusType;
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    acceptRanges?: string;
    date?: Date;
    isServerEncrypted?: boolean;
    encryptionKeySha256?: string;
    fileContentMD5?: Uint8Array;
    contentCrc64?: Uint8Array;
    /**
     * Specifies the encryption context to set on the file.
     */
    encryptionContext?: string;
    owner?: string;
    group?: string;
    permissions?: PathPermissions;
    /**
     * POSIX access control rights on files and directories.
     */
    acl: PathAccessControlItem[];
}

/** *********************************************************/
/** DataLakeFileClient option and response related models **/
/** *********************************************************/
export declare interface FileReadOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    rangeGetContentMD5?: boolean;
    rangeGetContentCrc64?: boolean;
    conditions?: DataLakeRequestConditions;
    onProgress?: (progress: TransferProgressEvent) => void;
    maxRetryRequests?: number;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
}

export declare type FileReadResponse = WithResponse<FileReadHeaders & {
    contentAsBlob?: Promise<Blob>;
    readableStreamBody?: NodeJS.ReadableStream;
}, FileReadHeaders>;

/**
 * Option interface for Data Lake file - readToBuffer operations
 *
 * See:
 * - {@link DataLakeFileClient.readToBuffer}
 */
export declare interface FileReadToBufferOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Access conditions headers.
     */
    conditions?: DataLakeRequestConditions;
    /**
     * Progress updater.
     */
    onProgress?: (progress: TransferProgressEvent) => void;
    /**
     * How many retries will perform for each read when the original chunk read stream ends unexpectedly.
     * Above kind of ends will not trigger retry policy defined in a pipeline,
     * because they doesn't emit network errors. Default value is 5.
     */
    maxRetryRequestsPerChunk?: number;
    /**
     * chunkSize is size of data every request trying to read.
     * Must be greater than or equal to 0, if set to 0 or undefined, it will automatically calculated according
     * to the file size.
     */
    chunkSize?: number;
    /**
     * Concurrency of parallel read.
     */
    concurrency?: number;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
}

/** Defines headers for Path_setExpiry operation. */
export declare interface FileSetExpiryHeaders {
    /** The ETag contains a value that you can use to perform operations conditionally. If the request version is 2011-08-18 or newer, the ETag value will be in quotes. */
    etag?: string;
    /** Returns the date and time the container was last modified. Any operation that modifies the blob, including an update of the blob's metadata or properties, changes the last-modified time of the blob. */
    lastModified?: Date;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the Blob service used to execute the request. This header is returned for requests made against version 2009-09-19 and above. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}

/**
 * Option interface for the {@link DataLakeFileClient.setExpiry} operation.
 */
export declare interface FileSetExpiryOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * The time to set the file to expire on, used in combination with the "Absolute" {@link FileExpiryMode}.
     * A time in the past is not allowed and milliseconds will be dropped.
     */
    expiresOn?: Date;
    /**
     * The number of milliseconds to elapse before the file expires, used in combination with the "RelativeToCreation" or "RelativeToNow" {@link FileExpiryMode}.
     */
    timeToExpireInMs?: number;
}

export declare type FileSetExpiryResponse = WithResponse<FileSetExpiryHeaders, FileSetExpiryHeaders>;

declare interface FileSystem_2 {
    name?: string;
    lastModified?: Date;
    etag?: string;
}

export declare interface FileSystemCreateHeaders {
    etag?: string;
    lastModified?: Date;
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
}

/** Defines headers for FileSystem_create operation. */
declare interface FileSystemCreateHeaders_2 {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the FileSystem. */
    etag?: string;
    /** The data and time the filesystem was last modified.  Operations on files and directories do not affect the last modified time. */
    lastModified?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    clientRequestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** A bool string indicates whether the namespace feature is enabled. If "true", the namespace is enabled for the filesystem. */
    namespaceEnabled?: string;
}

/**
 * Contains response data for the {@link DataLakeFileSystemClient.createIfNotExists} operation.
 */
export declare interface FileSystemCreateIfNotExistsResponse extends FileSystemCreateResponse {
    /**
     * Indicate whether the file system is successfully created. Is false when the file system is not changed as it already exists.
     */
    succeeded: boolean;
}

/** Optional parameters. */
declare interface FileSystemCreateOptionalParams extends coreClient.OperationOptions {
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Optional. User-defined properties to be stored with the filesystem, in the format of a comma-separated list of name and value pairs "n1=v1, n2=v2, ...", where each value is a base64 encoded string. Note that the string may only contain ASCII characters in the ISO-8859-1 character set.  If the filesystem exists, any properties not included in the list will be removed.  All properties are removed if the header is omitted.  To merge new and existing properties, first get all existing properties and the current E-Tag, then make a conditional request with the E-Tag and include values for all properties. */
    properties?: string;
}

/** **************************************************************/
/** DataLakeFileSystemClient option and response related models */
/** **************************************************************/
export declare interface FileSystemCreateOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    metadata?: Metadata;
    access?: PublicAccessType;
    /**
     * File System encryption scope info.
     */
    fileSystemEncryptionScope?: FileSystemEncryptionScope;
}

export declare type FileSystemCreateResponse = WithResponse<FileSystemCreateHeaders, FileSystemCreateHeaders>;

/** Contains response data for the create operation. */
declare type FileSystemCreateResponse_2 = FileSystemCreateHeaders_2;

export declare interface FileSystemDeleteHeaders {
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
}

/** Defines headers for FileSystem_delete operation. */
declare interface FileSystemDeleteHeaders_2 {
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
}

/**
 * Contains response data for the {@link DataLakeFileSystemClient.deleteIfExists} operation.
 */
export declare interface FileSystemDeleteIfExistsResponse extends FileSystemDeleteResponse {
    /**
     * Indicate whether the file system is successfully deleted. Is false if the file system doesn't exist in the first place.
     */
    succeeded: boolean;
}

/** Optional parameters. */
declare interface FileSystemDeleteOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
}

export declare interface FileSystemDeleteOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
}

export declare type FileSystemDeleteResponse = WithResponse<FileSystemDeleteHeaders, FileSystemDeleteHeaders>;

/** Contains response data for the delete operation. */
declare type FileSystemDeleteResponse_2 = FileSystemDeleteHeaders_2;

/**
 * Options to specify encryption scope on a file system.
 */
export declare interface FileSystemEncryptionScope {
    /** Optional.  Version 2021-02-12 and later.  Specifies the default encryption scope to set on the file system and use for all future writes. */
    defaultEncryptionScope?: string;
    /** Optional.  Version 2021-02-12 and newer.  If true, prevents any request from specifying a different encryption scope than the scope set on the container. */
    preventEncryptionScopeOverride?: boolean;
}

/**
 * Option interface for Data Lake file system exists operations
 *
 * See:
 * - {@link DataLakeFileSystemClient.exists}
 */
export declare interface FileSystemExistsOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
}

/**
 * Options to configure {@link DataLakeFileSystemClient.generateSasUrl} operation.
 */
export declare interface FileSystemGenerateSasUrlOptions extends CommonGenerateSasUrlOptions {
    /**
     * Optional only when identifier is provided. Specifies the list of permissions to be associated with the SAS.
     */
    permissions?: FileSystemSASPermissions;
}

export declare interface FileSystemGetAccessPolicyHeaders {
    publicAccess?: PublicAccessType;
    etag?: string;
    lastModified?: Date;
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
}

export declare interface FileSystemGetAccessPolicyOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: LeaseAccessConditions;
}

export declare type FileSystemGetAccessPolicyResponse = WithResponse<{
    signedIdentifiers: SignedIdentifier<AccessPolicy>[];
} & FileSystemGetAccessPolicyHeaders, FileSystemGetAccessPolicyHeaders, SignedIdentifier<RawAccessPolicy>[]>;

export declare interface FileSystemGetPropertiesHeaders {
    metadata?: Metadata;
    etag?: string;
    lastModified?: Date;
    leaseDuration?: LeaseDurationType;
    leaseState?: LeaseStateType;
    leaseStatus?: LeaseStatusType;
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
    publicAccess?: PublicAccessType;
    hasImmutabilityPolicy?: boolean;
    hasLegalHold?: boolean;
    /**
     * The default encryption scope for the file system.
     */
    defaultEncryptionScope?: string;
}

/** Defines headers for FileSystem_getProperties operation. */
declare interface FileSystemGetPropertiesHeaders_2 {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the filesystem.  Changes to filesystem properties affect the entity tag, but operations on files and directories do not. */
    etag?: string;
    /** The data and time the filesystem was last modified.  Changes to filesystem properties update the last modified time, but operations on files and directories do not. */
    lastModified?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** The user-defined properties associated with the filesystem.  A comma-separated list of name and value pairs in the format "n1=v1, n2=v2, ...", where each value is a base64 encoded string. Note that the string may only contain ASCII characters in the ISO-8859-1 character set. */
    properties?: string;
    /** A bool string indicates whether the namespace feature is enabled. If "true", the namespace is enabled for the filesystem. */
    namespaceEnabled?: string;
}

/** Optional parameters. */
declare interface FileSystemGetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
}

export declare interface FileSystemGetPropertiesOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: LeaseAccessConditions;
}

export declare type FileSystemGetPropertiesResponse = WithResponse<FileSystemGetPropertiesHeaders, FileSystemGetPropertiesHeaders>;

/** Contains response data for the getProperties operation. */
declare type FileSystemGetPropertiesResponse_2 = FileSystemGetPropertiesHeaders_2;

export declare interface FileSystemItem {
    name: string;
    properties: FileSystemProperties;
    metadata?: Metadata;
    deleted?: boolean;
    versionId?: string;
}

declare interface FileSystemList {
    filesystems?: FileSystem_2[];
}

/** Defines headers for FileSystem_listBlobHierarchySegment operation. */
export declare interface FileSystemListBlobHierarchySegmentHeaders {
    /** The media type of the body of the response. For List Blobs this is 'application/xml' */
    contentType?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** Indicates the version of the Blob service used to execute the request. This header is returned for requests made against version 2009-09-19 and above. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated */
    date?: Date;
    /** Error Code */
    errorCode?: string;
}

/** Optional parameters. */
declare interface FileSystemListBlobHierarchySegmentOptionalParams extends coreClient.OperationOptions {
    /** Filters results to filesystems within the specified prefix. */
    prefix?: string;
    /** An optional value that specifies the maximum number of items to return. If omitted or greater than 5,000, the response will include up to 5,000 items. */
    maxResults?: number;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** When the request includes this parameter, the operation returns a BlobPrefix element in the response body that acts as a placeholder for all blobs whose names begin with the same substring up to the appearance of the delimiter character. The delimiter may be a single character or a string. */
    delimiter?: string;
    /** A string value that identifies the portion of the list of containers to be returned with the next listing operation. The operation returns the NextMarker value within the response body if the listing operation did not return all containers remaining to be listed with the current page. The NextMarker value can be used as the value for the marker parameter in a subsequent call to request the next page of list items. The marker value is opaque to the client. */
    marker?: string;
    /** Include this parameter to specify one or more datasets to include in the response. */
    include?: ListBlobsIncludeItem[];
}

/** Contains response data for the listBlobHierarchySegment operation. */
declare type FileSystemListBlobHierarchySegmentResponse = FileSystemListBlobHierarchySegmentHeaders & ListBlobsHierarchySegmentResponse;

export declare type FileSystemListDeletedPathsResponse = WithResponse<DeletedPathList & FileSystemListBlobHierarchySegmentHeaders & ListBlobsHierarchySegmentResponse & {
    continuation?: string;
}, FileSystemListBlobHierarchySegmentHeaders, ListBlobsHierarchySegmentResponse>;

/** Defines headers for FileSystem_listPaths operation. */
export declare interface FileSystemListPathsHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the filesystem.  Changes to filesystem properties affect the entity tag, but operations on files and directories do not. */
    etag?: string;
    /** The data and time the filesystem was last modified.  Changes to filesystem properties update the last modified time, but operations on files and directories do not. */
    lastModified?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** If the number of paths to be listed exceeds the maxResults limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the list operation to continue listing the paths. */
    continuation?: string;
    /** Error Code */
    errorCode?: string;
}

/** Optional parameters. */
declare interface FileSystemListPathsOptionalParams extends coreClient.OperationOptions {
    /** Optional.  When deleting a directory, the number of paths that are deleted with each invocation is limited.  If the number of paths to be deleted exceeds this limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the delete operation to continue deleting the directory. */
    continuation?: string;
    /** An optional value that specifies the maximum number of items to return. If omitted or greater than 5,000, the response will include up to 5,000 items. */
    maxResults?: number;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Optional.  Filters results to paths within the specified directory. An error occurs if the directory does not exist. */
    path?: string;
    /** Optional. Valid only when Hierarchical Namespace is enabled for the account. If "true", the user identity values returned in the x-ms-owner, x-ms-group, and x-ms-acl response headers will be transformed from Azure Active Directory Object IDs to User Principal Names.  If "false", the values will be returned as Azure Active Directory Object IDs. The default value is false. Note that group and application Object IDs are not translated because they do not have unique friendly names. */
    upn?: boolean;
}

export declare type FileSystemListPathsResponse = WithResponse<PathList & FileSystemListPathsHeaders, FileSystemListPathsHeaders, PathListModel>;

/** Contains response data for the listPaths operation. */
declare type FileSystemListPathsResponse_2 = FileSystemListPathsHeaders & PathListModel;

/** Interface representing a FileSystemOperations. */
declare interface FileSystemOperations {
    /**
     * Create a FileSystem rooted at the specified location. If the FileSystem already exists, the
     * operation fails.  This operation does not support conditional HTTP requests.
     * @param options The options parameters.
     */
    create(options?: FileSystemCreateOptionalParams): Promise<FileSystemCreateResponse_2>;
    /**
     * Set properties for the FileSystem.  This operation supports conditional HTTP requests.  For more
     * information, see [Specifying Conditional Headers for Blob Service
     * Operations](https://docs.microsoft.com/en-us/rest/api/storageservices/specifying-conditional-headers-for-blob-service-operations).
     * @param options The options parameters.
     */
    setProperties(options?: FileSystemSetPropertiesOptionalParams): Promise<FileSystemSetPropertiesResponse>;
    /**
     * All system and user-defined filesystem properties are specified in the response headers.
     * @param options The options parameters.
     */
    getProperties(options?: FileSystemGetPropertiesOptionalParams): Promise<FileSystemGetPropertiesResponse_2>;
    /**
     * Marks the FileSystem for deletion.  When a FileSystem is deleted, a FileSystem with the same
     * identifier cannot be created for at least 30 seconds. While the filesystem is being deleted,
     * attempts to create a filesystem with the same identifier will fail with status code 409 (Conflict),
     * with the service returning additional error information indicating that the filesystem is being
     * deleted. All other operations, including operations on any files or directories within the
     * filesystem, will fail with status code 404 (Not Found) while the filesystem is being deleted. This
     * operation supports conditional HTTP requests.  For more information, see [Specifying Conditional
     * Headers for Blob Service
     * Operations](https://docs.microsoft.com/en-us/rest/api/storageservices/specifying-conditional-headers-for-blob-service-operations).
     * @param options The options parameters.
     */
    delete(options?: FileSystemDeleteOptionalParams): Promise<FileSystemDeleteResponse_2>;
    /**
     * List FileSystem paths and their properties.
     * @param recursive Required
     * @param options The options parameters.
     */
    listPaths(recursive: boolean, options?: FileSystemListPathsOptionalParams): Promise<FileSystemListPathsResponse_2>;
    /**
     * The List Blobs operation returns a list of the blobs under the specified container
     * @param options The options parameters.
     */
    listBlobHierarchySegment(options?: FileSystemListBlobHierarchySegmentOptionalParams): Promise<FileSystemListBlobHierarchySegmentResponse>;
}

export declare interface FileSystemProperties {
    lastModified: Date;
    etag: string;
    leaseStatus?: LeaseStatusType;
    leaseState?: LeaseStateType;
    leaseDuration?: LeaseDurationType;
    publicAccess?: PublicAccessType;
    hasImmutabilityPolicy?: boolean;
    hasLegalHold?: boolean;
    defaultEncryptionScope?: string;
    deletedOn?: Date;
    remainingRetentionDays?: number;
}

/**
 * Contains response data for the {@link DataLakeServiceClient.renameFileSystem} operation.
 */
export declare type FileSystemRenameResponse = ContainerRenameResponse;

/**
 * This is a helper class to construct a string representing the permissions granted by a ServiceSAS to a container.
 * Setting a value to true means that any SAS which uses these permissions will grant permissions for that operation.
 * Once all the values are set, this should be serialized with toString and set as the permissions field on a
 * {@link DataLakeSASSignatureValues} object. It is possible to construct the permissions string without this class, but
 * the order of the permissions is particular and this class guarantees correctness.
 */
export declare class FileSystemSASPermissions {
    /**
     * Creates an {@link FileSystemSASPermissions} from the specified permissions string. This method will throw an
     * Error if it encounters a character that does not correspond to a valid permission.
     *
     * @param permissions -
     */
    static parse(permissions: string): FileSystemSASPermissions;
    /**
     * Specifies Read access granted.
     */
    read: boolean;
    /**
     * Specifies Add access granted.
     */
    add: boolean;
    /**
     * Specifies Create access granted.
     */
    create: boolean;
    /**
     * Specifies Write access granted.
     */
    write: boolean;
    /**
     * Specifies Delete access granted.
     */
    delete: boolean;
    /**
     * Specifies List access granted.
     */
    list: boolean;
    /**
     * Specifies Move access granted.
     */
    move: boolean;
    /**
     * Specifies Execute access granted.
     */
    execute: boolean;
    /**
     * Specifies Ownership access granted, which allows the caller to set owner, owning group,
     * or act as the owner when renaming or deleting a blob (file or directory) within a folder
     * that has the sticky bit set.
     */
    manageOwnership: boolean;
    /**
     * Specifies Permission access granted, which allows the caller to set permissions and
     * POSIX ACLs on blobs (files and directories).
     */
    manageAccessControl: boolean;
    /**
     * Converts the given permissions to a string. Using this method will guarantee the permissions are in an
     * order accepted by the service.
     *
     * The order of the characters should be as specified here to ensure correctness.
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-a-service-sas
     *
     */
    toString(): string;
}

export declare interface FileSystemSetAccessPolicyHeaders {
    etag?: string;
    lastModified?: Date;
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
}

export declare interface FileSystemSetAccessPolicyOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
}

export declare type FileSystemSetAccessPolicyResponse = WithResponse<FileSystemSetAccessPolicyHeaders, FileSystemSetAccessPolicyHeaders>;

export declare interface FileSystemSetMetadataHeaders {
    etag?: string;
    lastModified?: Date;
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
}

export declare interface FileSystemSetMetadataOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
}

export declare type FileSystemSetMetadataResponse = WithResponse<FileSystemSetMetadataHeaders, FileSystemSetMetadataHeaders>;

/** Defines headers for FileSystem_setProperties operation. */
declare interface FileSystemSetPropertiesHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the filesystem.  Changes to filesystem properties affect the entity tag, but operations on files and directories do not. */
    etag?: string;
    /** The data and time the filesystem was last modified.  Changes to filesystem properties update the last modified time, but operations on files and directories do not. */
    lastModified?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
}

/** Optional parameters. */
declare interface FileSystemSetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Optional. User-defined properties to be stored with the filesystem, in the format of a comma-separated list of name and value pairs "n1=v1, n2=v2, ...", where each value is a base64 encoded string. Note that the string may only contain ASCII characters in the ISO-8859-1 character set.  If the filesystem exists, any properties not included in the list will be removed.  All properties are removed if the header is omitted.  To merge new and existing properties, first get all existing properties and the current E-Tag, then make a conditional request with the E-Tag and include values for all properties. */
    properties?: string;
}

/** Contains response data for the setProperties operation. */
declare type FileSystemSetPropertiesResponse = FileSystemSetPropertiesHeaders;

export declare interface FileSystemUndeletePathOption extends CommonOptions {
    abortSignal?: AbortSignalLike;
}

export declare type FileSystemUndeletePathResponse = WithResponse<PathUndeleteHeaders & {
    pathClient: DataLakePathClient;
}, PathUndeleteHeaders>;

/**
 * Contains response data for the {@link DataLakeServiceClient.undeleteFileSystem} operation.
 */
export declare type FileSystemUndeleteResponse = ContainerUndeleteResponse;

export declare type FileUploadResponse = WithResponse<PathFlushDataHeaders, PathFlushDataHeaders>;

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Generates a {@link SASQueryParameters} object which contains all SAS query parameters needed to make an actual
 * REST request.
 *
 * @see https://learn.microsoft.com/en-us/rest/api/storageservices/constructing-an-account-sas
 *
 * @param accountSASSignatureValues -
 * @param sharedKeyCredential -
 */
export declare function generateAccountSASQueryParameters(accountSASSignatureValues: AccountSASSignatureValues, sharedKeyCredential: StorageSharedKeyCredential): SASQueryParameters;

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Creates an instance of SASQueryParameters.
 *
 * Only accepts required settings needed to create a SAS. For optional settings please
 * set corresponding properties directly, such as permissions, startsOn and identifier.
 *
 * WARNING: When identifier is not provided, permissions and expiresOn are required.
 * You MUST assign value to identifier or expiresOn & permissions manually if you initial with
 * this constructor.
 *
 * Fill in the required details before running the following snippets.
 * @example
 * ```js
 * // Generate service level SAS for a file system
 * const containerSAS = generateDataLakeSASQueryParameters({
 *     fileSystemName, // Required
 *     permissions: ContainerSASPermissions.parse("racwdl"), // Required
 *     startsOn: new Date(), // Optional
 *     expiresOn: new Date(new Date().valueOf() + 86400), // Required. Date type
 *     ipRange: { start: "0.0.0.0", end: "255.255.255.255" }, // Optional
 *     protocol: SASProtocol.HttpsAndHttp, // Optional
 *     version: "2016-05-31" // Optional
 *   },
 *   sharedKeyCredential // StorageSharedKeyCredential - `new StorageSharedKeyCredential(account, accountKey)`
 * ).toString();
 * ```
 *
 * // Fill in the required details before running the snippet.
 * @example
 * ```js
 * // Generate service level SAS for a file
 * const fileSAS = generateDataLakeSASQueryParameters({
 *     fileSystemName, // Required
 *     fileName, // Required
 *     permissions: DataLakeSASPermissions.parse("racwd"), // Required
 *     startsOn: new Date(), // Optional
 *     expiresOn: new Date(new Date().valueOf() + 86400), // Required. Date type
 *     cacheControl: "cache-control-override", // Optional
 *     contentDisposition: "content-disposition-override", // Optional
 *     contentEncoding: "content-encoding-override", // Optional
 *     contentLanguage: "content-language-override", // Optional
 *     contentType: "content-type-override", // Optional
 *     ipRange: { start: "0.0.0.0", end: "255.255.255.255" }, // Optional
 *     protocol: SASProtocol.HttpsAndHttp, // Optional
 *     version: "2016-05-31" // Optional
 *   },
 *   sharedKeyCredential // StorageSharedKeyCredential - `new StorageSharedKeyCredential(account, accountKey)`
 * ).toString();
 * ```
 *
 * @param dataLakeSASSignatureValues -
 * @param sharedKeyCredential -
 */
export declare function generateDataLakeSASQueryParameters(dataLakeSASSignatureValues: DataLakeSASSignatureValues, sharedKeyCredential: StorageSharedKeyCredential): SASQueryParameters;

/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Creates an instance of SASQueryParameters.
 * WARNING: identifier will be ignored when generating user delegation SAS, permissions and expiresOn are required.
 *
 * @example
 * ```js
 * // Generate user delegation SAS for a file system
 * const userDelegationKey = await dataLakeServiceClient.getUserDelegationKey(startsOn, expiresOn);
 * const fileSystemSAS = generateDataLakeSASQueryParameters({
 *     fileSystemName, // Required
 *     permissions: FileSystemSASPermissions.parse("racwdl"), // Required
 *     startsOn, // Optional. Date type
 *     expiresOn, // Required. Date type
 *     ipRange: { start: "0.0.0.0", end: "255.255.255.255" }, // Optional
 *     protocol: SASProtocol.HttpsAndHttp, // Optional
 *     version: "2018-11-09" // Must greater than or equal to 2018-11-09 to generate user delegation SAS
 *   },
 *   userDelegationKey, // UserDelegationKey
 *   accountName
 * ).toString();
 * ```
 *
 * @param dataLakeSASSignatureValues -
 * @param userDelegationKey - Return value of `blobServiceClient.getUserDelegationKey()`
 * @param accountName -
 */
export declare function generateDataLakeSASQueryParameters(dataLakeSASSignatureValues: DataLakeSASSignatureValues, userDelegationKey: UserDelegationKey, accountName: string): SASQueryParameters;

/**
 * To get OAuth audience for a storage account for datalake service.
 */
export declare function getDataLakeServiceAccountAudience(storageAccountName: string): string;

export { HttpHeaders }

export { HttpOperationResponse }

export { HttpRequestBody }

export { isPipelineLike }

export { Lease }

export { LeaseAccessConditions }

/** Parameter group */
declare interface LeaseAccessConditions_2 {
    /** If specified, the operation only succeeds if the resource's lease is active and matches this ID. */
    leaseId?: string;
}

/** Defines values for LeaseAction. */
declare type LeaseAction = "acquire" | "auto-renew" | "release" | "acquire-release";

export declare type LeaseDurationType = "infinite" | "fixed";

export { LeaseOperationOptions }

export { LeaseOperationResponse }

export declare type LeaseStateType = "available" | "leased" | "expired" | "breaking" | "broken";

export declare type LeaseStatusType = "locked" | "unlocked";

/** An enumeration of blobs */
export declare interface ListBlobsHierarchySegmentResponse {
    serviceEndpoint: string;
    containerName: string;
    prefix?: string;
    marker?: string;
    maxResults?: number;
    delimiter?: string;
    segment: BlobHierarchyListSegment;
    nextMarker?: string;
}

/** Defines values for ListBlobsIncludeItem. */
declare type ListBlobsIncludeItem = "copy" | "deleted" | "metadata" | "snapshots" | "uncommittedblobs" | "versions" | "tags";

export declare interface ListDeletedPathsOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    /** Filters results to filesystems within the specified prefix. */
    prefix?: string;
}

export declare interface ListDeletedPathsSegmentOptions extends ListDeletedPathsOptions {
    maxResults?: number;
}

export declare interface ListFileSystemsSegmentResponse {
    serviceEndpoint: string;
    prefix?: string;
    marker?: string;
    maxPageSize?: number;
    fileSystemItems: FileSystemItem[];
    continuationToken?: string;
}

export declare interface ListPathsOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    recursive?: boolean;
    path?: string;
    userPrincipalName?: boolean;
}

export declare interface ListPathsSegmentOptions extends ListPathsOptions {
    maxResults?: number;
}

/**
 * The `@azure/logger` configuration for this package.
 */
export declare const logger: AzureLogger;

/** ********************************************************/
/** DataLakePathClient option and response related models */
/** ********************************************************/
export declare interface Metadata {
    [propertyName: string]: string;
}

export declare type ModifiedAccessConditions = Omit<ModifiedAccessConditions_3, "ifTags">;

/** Parameter group */
declare interface ModifiedAccessConditions_2 {
    /** Specify this header value to operate only on a blob if it has been modified since the specified date/time. */
    ifModifiedSince?: Date;
    /** Specify this header value to operate only on a blob if it has not been modified since the specified date/time. */
    ifUnmodifiedSince?: Date;
    /** Specify an ETag value to operate only on blobs with a matching value. */
    ifMatch?: string;
    /** Specify an ETag value to operate only on blobs without a matching value. */
    ifNoneMatch?: string;
}

/**
 * Creates a new Pipeline object with Credential provided.
 *
 * @param credential -  Such as AnonymousCredential, StorageSharedKeyCredential or any credential from the `@azure/identity` package to authenticate requests to the service. You can also provide an object that implements the TokenCredential interface. If not specified, AnonymousCredential is used.
 * @param pipelineOptions - Optional. Options.
 * @returns A new Pipeline object.
 */
export declare function newPipeline(credential?: StorageSharedKeyCredential | AnonymousCredential | TokenCredential, pipelineOptions?: StoragePipelineOptions): Pipeline;

export declare interface Path {
    name?: string;
    isDirectory?: boolean;
    lastModified?: Date;
    etag?: string;
    contentLength?: number;
    owner?: string;
    group?: string;
    permissions?: PathPermissions;
    /**
     * The name of the encryption scope under which the blob is encrypted.
     */
    encryptionScope?: string;
    /**
     * Creation time of the path.
     */
    createdOn?: Date;
    /**
     * Expiry time of the path.
     */
    expiresOn?: Date;
    /**
     * Specifies the encryption context to set on the file.
     */
    encryptionContext?: string;
}

export declare interface PathAccessControl {
    owner?: string;
    group?: string;
    permissions?: PathPermissions;
    /**
     * POSIX access control rights on files and directories.
     */
    acl: PathAccessControlItem[];
}

export declare interface PathAccessControlItem {
    /**
     * Indicates whether this is the default entry for the ACL.
     */
    defaultScope: boolean;
    /**
     * Specifies which role this entry targets.
     */
    accessControlType: AccessControlType;
    /**
     * Specifies the entity for which this entry applies.
     */
    entityId: string;
    /**
     * Access control permissions.
     */
    permissions: RolePermissions;
}

/** Defines headers for Path_appendData operation. */
export declare interface PathAppendDataHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** An HTTP entity tag associated with the file or directory. */
    etag?: string;
    /** If the blob has an MD5 hash and this operation is to read the full blob, this response header is returned so that the client can check for message content integrity. */
    contentMD5?: Uint8Array;
    /** This header is returned so that the client can check for message content integrity. The value of this header is computed by the Blob service; it is not necessarily the same value specified in the request headers. */
    xMsContentCrc64?: Uint8Array;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** The SHA-256 hash of the encryption key used to encrypt the blob. This header is only returned when the blob was encrypted with a customer-provided key. */
    encryptionKeySha256?: string;
    /** If the lease was auto-renewed with this request */
    leaseRenewed?: boolean;
}

/** Optional parameters. */
declare interface PathAppendDataOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions_2;
    /** Parameter group */
    cpkInfo?: CpkInfo;
    /** Parameter group */
    pathHttpHeaders?: PathHttpHeaders_2;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Proposed lease ID, in a GUID string format. The Blob service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. */
    proposedLeaseId?: string;
    /** The lease duration is required to acquire a lease, and specifies the duration of the lease in seconds.  The lease duration must be between 15 and 60 seconds or -1 for infinite lease. */
    leaseDuration?: number;
    /** This parameter allows the caller to upload data in parallel and control the order in which it is appended to the file.  It is required when uploading data to be appended to the file and when flushing previously uploaded data to the file.  The value must be the position where the data is to be appended.  Uploaded data is not immediately flushed, or written, to the file.  To flush, the previously uploaded data must be contiguous, the position parameter must be specified and equal to the length of the file after all data has been written, and there must not be a request entity body included with the request. */
    position?: number;
    /** Required for "Append Data" and "Flush Data".  Must be 0 for "Flush Data".  Must be the length of the request content in bytes for "Append Data". */
    contentLength?: number;
    /** Optional. If "acquire" it will acquire the lease. If "auto-renew" it will renew the lease. If "release" it will release the lease only on flush. If "acquire-release" it will acquire & complete the operation & release the lease once operation is done. */
    leaseAction?: LeaseAction;
    /** Specify the transactional crc64 for the body, to be validated by the service. */
    transactionalContentCrc64?: Uint8Array;
    /** If file should be flushed after the append */
    flush?: boolean;
}

/** Contains response data for the appendData operation. */
declare type PathAppendDataResponse = PathAppendDataHeaders;

/**
 * Options type for `setAccessControlRecursive`, `updateAccessControlRecursive` and `removeAccessControlRecursive`.
 */
export declare interface PathChangeAccessControlRecursiveOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Optional. If data set size exceeds batch size then operation will be split into multiple requests so that progress can be tracked.
     * Batch size should be between 1 and 2000. The default when unspecified is 2000.
     */
    batchSize?: number;
    /**
     * Optional. Defines maximum number of batches that single change Access Control operation can execute.
     * If maximum is reached before all subpaths are processed then continuation token can be used to resume operation.
     * Empty value indicates that maximum number of batches in unbound and operation continues till end.
     */
    maxBatches?: number;
    /**
     * Optional. Default false. If set to false, the operation will terminate quickly on encountering user failures.
     * If true, the operation will ignore user failures and proceed with the operation on other sub-entities of the directory.
     */
    continueOnFailure?: boolean;
    /**
     * Continuation token to continue next batch of operations.
     */
    continuationToken?: string;
    /**
     * Callback where caller can track progress of the operation
     * as well as collect paths that failed to change Access Control.
     */
    onProgress?: (progress: AccessControlChanges) => void;
}

/**
 * Response type for `setAccessControlRecursive`, `updateAccessControlRecursive` and `removeAccessControlRecursive`.
 */
export declare interface PathChangeAccessControlRecursiveResponse {
    /**
     * Contains counts of paths changed from start of the operation.
     */
    counters: AccessControlChangeCounters;
    /**
     * Optional. Value is present when operation is split into multiple batches and can be used to resume progress.
     */
    continuationToken?: string;
}

/** Defines headers for Path_create operation. */
export declare interface PathCreateHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the file or directory. */
    etag?: string;
    /** The data and time the file or directory was last modified.  Write operations on the file or directory update the last modified time. */
    lastModified?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** When renaming a directory, the number of paths that are renamed with each invocation is limited.  If the number of paths to be renamed exceeds this limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the rename operation to continue renaming the directory. */
    continuation?: string;
    /** The size of the resource in bytes. */
    contentLength?: number;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** The SHA-256 hash of the encryption key used to encrypt the blob. This header is only returned when the blob was encrypted with a customer-provided key. */
    encryptionKeySha256?: string;
    /** Error Code */
    errorCode?: string;
}

export declare interface PathCreateHttpHeaders {
    cacheControl?: string;
    contentEncoding?: string;
    contentLanguage?: string;
    contentDisposition?: string;
    contentType?: string;
}

export declare interface PathCreateIfNotExistsOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    metadata?: Metadata;
    permissions?: string;
    umask?: string;
    /**
     * Optional. The owner of the blob or directory.
     */
    owner?: string;
    /**
     * Optional. The owning group of the blob or directory.
     */
    group?: string;
    /**
     * Optional. POSIX access control rights on files and directories.
     */
    acl?: PathAccessControlItem[];
    pathHttpHeaders?: PathCreateHttpHeaders;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
    /**
     * Proposed lease ID, in a GUID string format. The Blob service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats.
     */
    proposedLeaseId?: string;
    /**
     * The lease duration is required to acquire a lease, and specifies the duration of the lease in seconds.  The lease duration must be between 15 and 60 seconds or -1 for infinite lease.
     */
    leaseDuration?: number;
    /**
     * Optional. Options for scheduling the deletion of a path.
     * A number value indicates duration before file should be deleted in milliseconds.
     * A Date value indicates the time to set for when the path will be deleted.
     * Does not apply to directories.
     */
    expiresOn?: number | Date;
    /**
     * Optional. Specifies the encryption context to set on the file.
     */
    encryptionContext?: string;
}

/**
 * Contains response data for the {@link DataLakePathClient.createIfNotExists} operation.
 */
export declare interface PathCreateIfNotExistsResponse extends PathCreateResponse {
    /**
     * Indicate whether the directory/file is successfully created. Is false when the directory/file is not changed as it already exists.
     */
    succeeded: boolean;
}

/** Optional parameters. */
declare interface PathCreateOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Parameter group */
    pathHttpHeaders?: PathHttpHeaders_2;
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions_2;
    /** Parameter group */
    sourceModifiedAccessConditions?: SourceModifiedAccessConditions;
    /** Parameter group */
    cpkInfo?: CpkInfo;
    /** Optional.  When deleting a directory, the number of paths that are deleted with each invocation is limited.  If the number of paths to be deleted exceeds this limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the delete operation to continue deleting the directory. */
    continuation?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Optional. User-defined properties to be stored with the filesystem, in the format of a comma-separated list of name and value pairs "n1=v1, n2=v2, ...", where each value is a base64 encoded string. Note that the string may only contain ASCII characters in the ISO-8859-1 character set.  If the filesystem exists, any properties not included in the list will be removed.  All properties are removed if the header is omitted.  To merge new and existing properties, first get all existing properties and the current E-Tag, then make a conditional request with the E-Tag and include values for all properties. */
    properties?: string;
    /** Required only for Create File and Create Directory. The value must be "file" or "directory". */
    resource?: PathResourceTypeModel;
    /** Optional. Valid only when namespace is enabled. This parameter determines the behavior of the rename operation. The value must be "legacy" or "posix", and the default value will be "posix". */
    mode?: PathRenameModeModel;
    /** An optional file or directory to be renamed.  The value must have the following format: "/{filesystem}/{path}".  If "x-ms-properties" is specified, the properties will overwrite the existing properties; otherwise, the existing properties will be preserved. This value must be a URL percent-encoded string. Note that the string may only contain ASCII characters in the ISO-8859-1 character set. */
    renameSource?: string;
    /** A lease ID for the source path. If specified, the source path must have an active lease and the lease ID must match. */
    sourceLeaseId?: string;
    /** Optional and only valid if Hierarchical Namespace is enabled for the account. Sets POSIX access permissions for the file owner, the file owning group, and others. Each class may be granted read, write, or execute permission.  The sticky bit is also supported.  Both symbolic (rwxrw-rw-) and 4-digit octal notation (e.g. 0766) are supported. */
    permissions?: string;
    /** Optional and only valid if Hierarchical Namespace is enabled for the account. When creating a file or directory and the parent folder does not have a default ACL, the umask restricts the permissions of the file or directory to be created.  The resulting permission is given by p bitwise and not u, where p is the permission and u is the umask.  For example, if p is 0777 and u is 0057, then the resulting permission is 0720.  The default permission is 0777 for a directory and 0666 for a file.  The default umask is 0027.  The umask must be specified in 4-digit octal notation (e.g. 0766). */
    umask?: string;
    /** Optional. The owner of the blob or directory. */
    owner?: string;
    /** Optional. The owning group of the blob or directory. */
    group?: string;
    /** Sets POSIX access control rights on files and directories. The value is a comma-separated list of access control entries. Each access control entry (ACE) consists of a scope, a type, a user or group identifier, and permissions in the format "[scope:][type]:[id]:[permissions]". */
    acl?: string;
    /** Proposed lease ID, in a GUID string format. The Blob service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. */
    proposedLeaseId?: string;
    /** The lease duration is required to acquire a lease, and specifies the duration of the lease in seconds.  The lease duration must be between 15 and 60 seconds or -1 for infinite lease. */
    leaseDuration?: number;
    /** Required. Indicates mode of the expiry time */
    expiryOptions?: FileExpiryMode;
    /** The time to set the blob to expiry */
    expiresOn?: string;
    /** Specifies the encryption context to set on the file. */
    encryptionContext?: string;
}

export declare interface PathCreateOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    metadata?: Metadata;
    permissions?: string;
    umask?: string;
    /**
     * Optional. The owner of the blob or directory.
     */
    owner?: string;
    /**
     * Optional. The owning group of the blob or directory.
     */
    group?: string;
    /**
     * Optional. POSIX access control rights on files and directories.
     */
    acl?: PathAccessControlItem[];
    conditions?: DataLakeRequestConditions;
    pathHttpHeaders?: PathCreateHttpHeaders;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
    /**
     * Proposed lease ID, in a GUID string format. The Blob service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats.
     */
    proposedLeaseId?: string;
    /**
     * The lease duration is required to acquire a lease, and specifies the duration of the lease in seconds.  The lease duration must be between 15 and 60 seconds or -1 for infinite lease.
     */
    leaseDuration?: number;
    /**
     * Optional. Options for scheduling the deletion of a path.
     * A number value indicates duration before file should be deleted in milliseconds.
     * A Date value indicates the time to set for when the path will be deleted.
     * Does not apply to directories.
     */
    expiresOn?: number | Date;
    /**
     * Optional. Specifies the encryption context to set on the file.
     */
    encryptionContext?: string;
}

export declare type PathCreateResponse = WithResponse<PathCreateHeaders, PathCreateHeaders>;

/** Contains response data for the create operation. */
declare type PathCreateResponse_2 = PathCreateHeaders;

/** Defines headers for Path_delete operation. */
export declare interface PathDeleteHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** When deleting a directory, the number of paths that are deleted with each invocation is limited.  If the number of paths to be deleted exceeds this limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the delete operation to continue deleting the directory. */
    continuation?: string;
    /** Returned only for hierarchical namespace space enabled accounts when soft delete is enabled. A unique identifier for the entity that can be used to restore it. See the Undelete REST API for more information. */
    deletionId?: string;
    /** Error Code */
    errorCode?: string;
}

/**
 * Contains response data for the {@link DataLakePathClient.deleteIfExists} operation.
 */
export declare interface PathDeleteIfExistsResponse extends PathDeleteResponse {
    /**
     * Indicate whether the directory/file is successfully deleted. Is false if the directory/file doesn't exist in the first place.
     */
    succeeded: boolean;
}

/** Optional parameters. */
declare interface PathDeleteOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions_2;
    /** Optional.  When deleting a directory, the number of paths that are deleted with each invocation is limited.  If the number of paths to be deleted exceeds this limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the delete operation to continue deleting the directory. */
    continuation?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Required */
    recursive?: boolean;
    /** If true, paginated behavior will be seen. Pagination is for the recursive ACL checks as a POSIX requirement in the server and Delete in an atomic operation once the ACL checks are completed. If false or missing, normal default behavior will kick in, which may timeout in case of very large directories due to recursive ACL checks. This new parameter is introduced for backward compatibility. */
    paginated?: boolean;
}

export declare interface PathDeleteOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
}

export declare type PathDeleteResponse = WithResponse<PathDeleteHeaders, PathDeleteHeaders>;

/** Contains response data for the delete operation. */
declare type PathDeleteResponse_2 = PathDeleteHeaders;

/**
 * Option interface for Data Lake directory/file exists operations
 *
 * See:
 * - {@link DataLakePathClient.exists}
 */
export declare interface PathExistsOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
}

/** Defines headers for Path_flushData operation. */
export declare interface PathFlushDataHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the file or directory. */
    etag?: string;
    /** The data and time the file or directory was last modified.  Write operations on the file or directory update the last modified time. */
    lastModified?: Date;
    /** The size of the resource in bytes. */
    contentLength?: number;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** The SHA-256 hash of the encryption key used to encrypt the blob. This header is only returned when the blob was encrypted with a customer-provided key. */
    encryptionKeySha256?: string;
    /** If the lease was auto-renewed with this request */
    leaseRenewed?: boolean;
}

/** Optional parameters. */
declare interface PathFlushDataOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Parameter group */
    pathHttpHeaders?: PathHttpHeaders_2;
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions_2;
    /** Parameter group */
    cpkInfo?: CpkInfo;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Proposed lease ID, in a GUID string format. The Blob service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. */
    proposedLeaseId?: string;
    /** The lease duration is required to acquire a lease, and specifies the duration of the lease in seconds.  The lease duration must be between 15 and 60 seconds or -1 for infinite lease. */
    leaseDuration?: number;
    /** This parameter allows the caller to upload data in parallel and control the order in which it is appended to the file.  It is required when uploading data to be appended to the file and when flushing previously uploaded data to the file.  The value must be the position where the data is to be appended.  Uploaded data is not immediately flushed, or written, to the file.  To flush, the previously uploaded data must be contiguous, the position parameter must be specified and equal to the length of the file after all data has been written, and there must not be a request entity body included with the request. */
    position?: number;
    /** Valid only for flush operations.  If "true", uncommitted data is retained after the flush operation completes; otherwise, the uncommitted data is deleted after the flush operation.  The default is false.  Data at offsets less than the specified position are written to the file when flush succeeds, but this optional parameter allows data after the flush position to be retained for a future flush operation. */
    retainUncommittedData?: boolean;
    /** Azure Storage Events allow applications to receive notifications when files change. When Azure Storage Events are enabled, a file changed event is raised. This event has a property indicating whether this is the final change to distinguish the difference between an intermediate flush to a file stream and the final close of a file stream. The close query parameter is valid only when the action is "flush" and change notifications are enabled. If the value of close is "true" and the flush operation completes successfully, the service raises a file change notification with a property indicating that this is the final update (the file stream has been closed). If "false" a change notification is raised indicating the file has changed. The default is false. This query parameter is set to true by the Hadoop ABFS driver to indicate that the file stream has been closed." */
    close?: boolean;
    /** Required for "Append Data" and "Flush Data".  Must be 0 for "Flush Data".  Must be the length of the request content in bytes for "Append Data". */
    contentLength?: number;
    /** Optional. If "acquire" it will acquire the lease. If "auto-renew" it will renew the lease. If "release" it will release the lease only on flush. If "acquire-release" it will acquire & complete the operation & release the lease once operation is done. */
    leaseAction?: LeaseAction;
}

/** Contains response data for the flushData operation. */
declare type PathFlushDataResponse = PathFlushDataHeaders;

export declare interface PathGetAccessControlHeaders {
    date?: Date;
    etag?: string;
    lastModified?: Date;
    owner?: string;
    group?: string;
    requestId?: string;
    version?: string;
}

export declare interface PathGetAccessControlOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
    userPrincipalName?: boolean;
}

export declare type PathGetAccessControlResponse = WithResponse<PathAccessControl & PathGetAccessControlHeaders, PathGetPropertiesHeadersModel>;

/**
 * Defines values for PathGetPropertiesAction.
 * Possible values include: 'getAccessControl', 'getStatus'
 * @readonly
 */
export declare enum PathGetPropertiesAction {
    GetAccessControl = "getAccessControl",
    GetStatus = "getStatus"
}

/** Defines values for PathGetPropertiesAction. */
export declare type PathGetPropertiesActionModel = "getAccessControl" | "getStatus";

export declare interface PathGetPropertiesHeaders {
    lastModified?: Date;
    createdOn?: Date;
    metadata?: Metadata;
    copyCompletedOn?: Date;
    copyStatusDescription?: string;
    copyId?: string;
    copyProgress?: string;
    copySource?: string;
    copyStatus?: CopyStatusType;
    isIncrementalCopy?: boolean;
    destinationSnapshot?: string;
    leaseDuration?: LeaseDurationType;
    leaseState?: LeaseStateType;
    leaseStatus?: LeaseStatusType;
    contentLength?: number;
    contentType?: string;
    etag?: string;
    contentMD5?: Uint8Array;
    contentEncoding?: string;
    contentDisposition?: string;
    contentLanguage?: string;
    cacheControl?: string;
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
    acceptRanges?: string;
    isServerEncrypted?: boolean;
    encryptionKeySha256?: string;
    /**
     * Returns the name of the encryption scope used to encrypt the path contents and application metadata.
     * Note that the absence of this header implies use of the default account encryption scope.
     */
    encryptionScope?: string;
    accessTier?: string;
    accessTierInferred?: boolean;
    archiveStatus?: string;
    accessTierChangedOn?: Date;
    /**
     * The time the file will expire.
     */
    expiresOn?: Date;
    /**
     * Optional. Specifies the encryption context to set on the file.
     */
    encryptionContext?: string;
    owner?: string;
    group?: string;
    permissions?: PathPermissions;
    /**
     * POSIX access control rights on files and directories.
     */
    acl: PathAccessControlItem[];
}

/** Defines headers for Path_getProperties operation. */
export declare interface PathGetPropertiesHeadersModel {
    /** Indicates that the service supports requests for partial file content. */
    acceptRanges?: string;
    /** If the Cache-Control request header has previously been set for the resource, that value is returned in this header. */
    cacheControl?: string;
    /** If the Content-Disposition request header has previously been set for the resource, that value is returned in this header. */
    contentDisposition?: string;
    /** If the Content-Encoding request header has previously been set for the resource, that value is returned in this header. */
    contentEncoding?: string;
    /** If the Content-Language request header has previously been set for the resource, that value is returned in this header. */
    contentLanguage?: string;
    /** The size of the resource in bytes. */
    contentLength?: number;
    /** Indicates the range of bytes returned in the event that the client requested a subset of the file by setting the Range request header. */
    contentRange?: string;
    /** The content type specified for the resource. If no content type was specified, the default content type is application/octet-stream. */
    contentType?: string;
    /** The MD5 hash of complete file stored in storage. This header is returned only for "GetProperties" operation. If the Content-MD5 header has been set for the file, this response header is returned for GetProperties call so that the client can check for message content integrity. */
    contentMD5?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the file or directory. */
    etag?: string;
    /** The data and time the file or directory was last modified.  Write operations on the file or directory update the last modified time. */
    lastModified?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** The type of the resource.  The value may be "file" or "directory".  If not set, the value is "file". */
    resourceType?: string;
    /** The user-defined properties associated with the file or directory, in the format of a comma-separated list of name and value pairs "n1=v1, n2=v2, ...", where each value is a base64 encoded string. Note that the string may only contain ASCII characters in the ISO-8859-1 character set. */
    properties?: string;
    /** The owner of the file or directory. Included in the response if Hierarchical Namespace is enabled for the account. */
    owner?: string;
    /** The owning group of the file or directory. Included in the response if Hierarchical Namespace is enabled for the account. */
    group?: string;
    /** The POSIX access permissions for the file owner, the file owning group, and others. Included in the response if Hierarchical Namespace is enabled for the account. */
    permissions?: string;
    /** The POSIX access control list for the file or directory.  Included in the response only if the action is "getAccessControl" and Hierarchical Namespace is enabled for the account. */
    acl?: string;
    /** When a resource is leased, specifies whether the lease is of infinite or fixed duration. */
    leaseDuration?: string;
    /** Lease state of the resource. */
    leaseState?: string;
    /** The lease status of the resource. */
    leaseStatus?: string;
    /** Error Code */
    errorCode?: string;
}

/** Optional parameters. */
declare interface PathGetPropertiesOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions_2;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Optional. Valid only when Hierarchical Namespace is enabled for the account. If "true", the user identity values returned in the x-ms-owner, x-ms-group, and x-ms-acl response headers will be transformed from Azure Active Directory Object IDs to User Principal Names.  If "false", the values will be returned as Azure Active Directory Object IDs. The default value is false. Note that group and application Object IDs are not translated because they do not have unique friendly names. */
    upn?: boolean;
    /** Optional. If the value is "getStatus" only the system defined properties for the path are returned. If the value is "getAccessControl" the access control list is returned in the response headers (Hierarchical Namespace must be enabled for the account), otherwise the properties are returned. */
    action?: PathGetPropertiesActionModel;
}

export declare interface PathGetPropertiesOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
}

export declare type PathGetPropertiesResponse = WithResponse<PathGetPropertiesHeaders, PathGetPropertiesHeaders>;

/** Contains response data for the getProperties operation. */
declare type PathGetPropertiesResponse_2 = PathGetPropertiesHeadersModel;

export declare interface PathHttpHeaders {
    cacheControl?: string;
    contentEncoding?: string;
    contentLanguage?: string;
    contentDisposition?: string;
    contentType?: string;
    contentMD5?: Uint8Array;
}

/** Parameter group */
declare interface PathHttpHeaders_2 {
    /** Optional. Sets the blob's cache control. If specified, this property is stored with the blob and returned with a read request. */
    cacheControl?: string;
    /** Optional. Sets the blob's content encoding. If specified, this property is stored with the blob and returned with a read request. */
    contentEncoding?: string;
    /** Optional. Set the blob's content language. If specified, this property is stored with the blob and returned with a read request. */
    contentLanguage?: string;
    /** Optional. Sets the blob's Content-Disposition header. */
    contentDisposition?: string;
    /** Optional. Sets the blob's content type. If specified, this property is stored with the blob and returned with a read request. */
    contentType?: string;
    /** Specify the transactional md5 for the body, to be validated by the service. */
    contentMD5?: Uint8Array;
    /** Specify the transactional md5 for the body, to be validated by the service. */
    transactionalContentHash?: Uint8Array;
}

/** Defines values for PathLeaseAction. */
declare type PathLeaseAction = "acquire" | "break" | "change" | "renew" | "release";

/** Defines headers for Path_lease operation. */
declare interface PathLeaseHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the file. */
    etag?: string;
    /** The data and time the file was last modified.  Write operations on the file update the last modified time. */
    lastModified?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** A successful "renew" action returns the lease ID. */
    leaseId?: string;
}

/** Optional parameters. */
declare interface PathLeaseOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions_2;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Proposed lease ID, in a GUID string format. The Blob service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. */
    proposedLeaseId?: string;
    /** The lease break period duration is optional to break a lease, and  specifies the break period of the lease in seconds.  The lease break  duration must be between 0 and 60 seconds. */
    xMsLeaseBreakPeriod?: number;
}

/** Contains response data for the lease operation. */
declare type PathLeaseResponse = PathLeaseHeaders;

export declare interface PathList {
    pathItems?: Path[];
}

export declare interface PathListModel {
    paths?: PathModel[];
}

export declare interface PathModel {
    name?: string;
    isDirectory?: boolean;
    lastModified?: Date;
    etag?: string;
    contentLength?: number;
    owner?: string;
    group?: string;
    permissions?: string;
    /** The name of the encryption scope under which the blob is encrypted. */
    encryptionScope?: string;
    creationTime?: string;
    expiryTime?: string;
    encryptionContext?: string;
}

export declare interface PathMoveOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
    destinationConditions?: DataLakeRequestConditions;
}

export declare type PathMoveResponse = WithResponse<PathRemoveHeaders, PathRemoveHeaders>;

/** Interface representing a PathOperations. */
declare interface PathOperations {
    /**
     * Create or rename a file or directory.    By default, the destination is overwritten and if the
     * destination already exists and has a lease the lease is broken.  This operation supports conditional
     * HTTP requests.  For more information, see [Specifying Conditional Headers for Blob Service
     * Operations](https://docs.microsoft.com/en-us/rest/api/storageservices/specifying-conditional-headers-for-blob-service-operations).
     *  To fail if the destination already exists, use a conditional request with If-None-Match: "*".
     * @param options The options parameters.
     */
    create(options?: PathCreateOptionalParams): Promise<PathCreateResponse_2>;
    /**
     * Uploads data to be appended to a file, flushes (writes) previously uploaded data to a file, sets
     * properties for a file or directory, or sets access control for a file or directory. Data can only be
     * appended to a file. Concurrent writes to the same file using multiple clients are not supported.
     * This operation supports conditional HTTP requests. For more information, see [Specifying Conditional
     * Headers for Blob Service
     * Operations](https://docs.microsoft.com/en-us/rest/api/storageservices/specifying-conditional-headers-for-blob-service-operations).
     * @param action The action must be "append" to upload data to be appended to a file, "flush" to flush
     *               previously uploaded data to a file, "setProperties" to set the properties of a file or directory,
     *               "setAccessControl" to set the owner, group, permissions, or access control list for a file or
     *               directory, or  "setAccessControlRecursive" to set the access control list for a directory
     *               recursively. Note that Hierarchical Namespace must be enabled for the account in order to use access
     *               control.  Also note that the Access Control List (ACL) includes permissions for the owner, owning
     *               group, and others, so the x-ms-permissions and x-ms-acl request headers are mutually exclusive.
     * @param mode Mode "set" sets POSIX access control rights on files and directories, "modify" modifies
     *             one or more POSIX access control rights  that pre-exist on files and directories, "remove" removes
     *             one or more POSIX access control rights  that were present earlier on files and directories
     * @param body Initial data
     * @param options The options parameters.
     */
    update(action: PathUpdateAction, mode: PathSetAccessControlRecursiveMode, body: coreRestPipeline.RequestBodyType, options?: PathUpdateOptionalParams): Promise<PathUpdateResponse>;
    /**
     * Create and manage a lease to restrict write and delete access to the path. This operation supports
     * conditional HTTP requests.  For more information, see [Specifying Conditional Headers for Blob
     * Service
     * Operations](https://docs.microsoft.com/en-us/rest/api/storageservices/specifying-conditional-headers-for-blob-service-operations).
     * @param xMsLeaseAction There are five lease actions: "acquire", "break", "change", "renew", and
     *                       "release". Use "acquire" and specify the "x-ms-proposed-lease-id" and "x-ms-lease-duration" to
     *                       acquire a new lease. Use "break" to break an existing lease. When a lease is broken, the lease break
     *                       period is allowed to elapse, during which time no lease operation except break and release can be
     *                       performed on the file. When a lease is successfully broken, the response indicates the interval in
     *                       seconds until a new lease can be acquired. Use "change" and specify the current lease ID in
     *                       "x-ms-lease-id" and the new lease ID in "x-ms-proposed-lease-id" to change the lease ID of an active
     *                       lease. Use "renew" and specify the "x-ms-lease-id" to renew an existing lease. Use "release" and
     *                       specify the "x-ms-lease-id" to release a lease.
     * @param options The options parameters.
     */
    lease(xMsLeaseAction: PathLeaseAction, options?: PathLeaseOptionalParams): Promise<PathLeaseResponse>;
    /**
     * Read the contents of a file.  For read operations, range requests are supported. This operation
     * supports conditional HTTP requests.  For more information, see [Specifying Conditional Headers for
     * Blob Service
     * Operations](https://docs.microsoft.com/en-us/rest/api/storageservices/specifying-conditional-headers-for-blob-service-operations).
     * @param options The options parameters.
     */
    read(options?: PathReadOptionalParams): Promise<PathReadResponse>;
    /**
     * Get Properties returns all system and user defined properties for a path. Get Status returns all
     * system defined properties for a path. Get Access Control List returns the access control list for a
     * path. This operation supports conditional HTTP requests.  For more information, see [Specifying
     * Conditional Headers for Blob Service
     * Operations](https://docs.microsoft.com/en-us/rest/api/storageservices/specifying-conditional-headers-for-blob-service-operations).
     * @param options The options parameters.
     */
    getProperties(options?: PathGetPropertiesOptionalParams): Promise<PathGetPropertiesResponse_2>;
    /**
     * Delete the file or directory. This operation supports conditional HTTP requests.  For more
     * information, see [Specifying Conditional Headers for Blob Service
     * Operations](https://docs.microsoft.com/en-us/rest/api/storageservices/specifying-conditional-headers-for-blob-service-operations).
     * @param options The options parameters.
     */
    delete(options?: PathDeleteOptionalParams): Promise<PathDeleteResponse_2>;
    /**
     * Set the owner, group, permissions, or access control list for a path.
     * @param options The options parameters.
     */
    setAccessControl(options?: PathSetAccessControlOptionalParams): Promise<PathSetAccessControlResponse_2>;
    /**
     * Set the access control list for a path and sub-paths.
     * @param mode Mode "set" sets POSIX access control rights on files and directories, "modify" modifies
     *             one or more POSIX access control rights  that pre-exist on files and directories, "remove" removes
     *             one or more POSIX access control rights  that were present earlier on files and directories
     * @param options The options parameters.
     */
    setAccessControlRecursive(mode: PathSetAccessControlRecursiveMode, options?: PathSetAccessControlRecursiveOptionalParams): Promise<PathSetAccessControlRecursiveResponse>;
    /**
     * Set the owner, group, permissions, or access control list for a path.
     * @param options The options parameters.
     */
    flushData(options?: PathFlushDataOptionalParams): Promise<PathFlushDataResponse>;
    /**
     * Append data to the file.
     * @param body Initial data
     * @param options The options parameters.
     */
    appendData(body: coreRestPipeline.RequestBodyType, options?: PathAppendDataOptionalParams): Promise<PathAppendDataResponse>;
    /**
     * Sets the time a blob will expire and be deleted.
     * @param expiryOptions Required. Indicates mode of the expiry time
     * @param options The options parameters.
     */
    setExpiry(expiryOptions: FileExpiryMode, options?: PathSetExpiryOptionalParams): Promise<PathSetExpiryResponse>;
    /**
     * Undelete a path that was previously soft deleted
     * @param options The options parameters.
     */
    undelete(options?: PathUndeleteOptionalParams): Promise<PathUndeleteResponse>;
}

export declare interface PathPermissions {
    owner: RolePermissions;
    group: RolePermissions;
    other: RolePermissions;
    stickyBit: boolean;
    extendedAcls: boolean;
}

/** Defines headers for Path_read operation. */
declare interface PathReadHeaders {
    /** Indicates that the service supports requests for partial file content. */
    acceptRanges?: string;
    /** If the Cache-Control request header has previously been set for the resource, that value is returned in this header. */
    cacheControl?: string;
    /** If the Content-Disposition request header has previously been set for the resource, that value is returned in this header. */
    contentDisposition?: string;
    /** If the Content-Encoding request header has previously been set for the resource, that value is returned in this header. */
    contentEncoding?: string;
    /** If the Content-Language request header has previously been set for the resource, that value is returned in this header. */
    contentLanguage?: string;
    /** The size of the resource in bytes. */
    contentLength?: number;
    /** Indicates the range of bytes returned in the event that the client requested a subset of the file by setting the Range request header. */
    contentRange?: string;
    /** The content type specified for the resource. If no content type was specified, the default content type is application/octet-stream. */
    contentType?: string;
    /** The MD5 hash of complete file. If the file has an MD5 hash and this read operation is to read the complete file, this response header is returned so that the client can check for message content integrity. */
    contentMD5?: string;
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the file or directory. */
    etag?: string;
    /** The data and time the file or directory was last modified.  Write operations on the file or directory update the last modified time. */
    lastModified?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** The type of the resource.  The value may be "file" or "directory".  If not set, the value is "file". */
    resourceType?: string;
    /** The user-defined properties associated with the file or directory, in the format of a comma-separated list of name and value pairs "n1=v1, n2=v2, ...", where each value is a base64 encoded string. Note that the string may only contain ASCII characters in the ISO-8859-1 character set. */
    properties?: string;
    /** When a resource is leased, specifies whether the lease is of infinite or fixed duration. */
    leaseDuration?: string;
    /** Lease state of the resource. */
    leaseState?: string;
    /** The lease status of the resource. */
    leaseStatus?: string;
    /** The value of this header is set to true if the contents of the request are successfully encrypted using the specified algorithm, and false otherwise. */
    isServerEncrypted?: boolean;
    /** The SHA-256 hash of the encryption key used to encrypt the blob. This header is only returned when the blob was encrypted with a customer-provided key. */
    encryptionKeySha256?: string;
}

/** Optional parameters. */
declare interface PathReadOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions_2;
    /** Parameter group */
    cpkInfo?: CpkInfo;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** The HTTP Range request header specifies one or more byte ranges of the resource to be retrieved. */
    range?: string;
    /** Optional. When this header is set to "true" and specified together with the Range header, the service returns the MD5 hash for the range, as long as the range is less than or equal to 4MB in size. If this header is specified without the Range header, the service returns status code 400 (Bad Request). If this header is set to true when the range exceeds 4 MB in size, the service returns status code 400 (Bad Request). */
    xMsRangeGetContentMd5?: boolean;
}

/** Contains response data for the read operation. */
declare type PathReadResponse = PathReadHeaders & {
    /**
     * BROWSER ONLY
     *
     * The response body as a browser Blob.
     * Always `undefined` in node.js.
     */
    blobBody?: Promise<Blob>;
    /**
     * NODEJS ONLY
     *
     * The response body as a node.js Readable stream.
     * Always `undefined` in the browser.
     */
    readableStreamBody?: NodeJS.ReadableStream;
};

export declare interface PathRemoveHeaders {
    date?: Date;
    etag?: string;
    lastModified?: Date;
    requestId?: string;
    version?: string;
    contentLength?: number;
}

/**
 * Defines values for PathRenameMode.
 * Possible values include: 'legacy', 'posix'
 * @readonly
 */
export declare enum PathRenameMode {
    Legacy = "legacy",
    Posix = "posix"
}

/** Defines values for PathRenameMode. */
export declare type PathRenameModeModel = "legacy" | "posix";

/**
 * Defines values for PathResourceType.
 * Possible values include: 'directory', 'file'
 * @readonly
 */
export declare enum PathResourceType {
    Directory = "directory",
    File = "file"
}

/** Defines values for PathResourceType. */
export declare type PathResourceTypeModel = "directory" | "file";

/** Defines headers for Path_setAccessControl operation. */
export declare interface PathSetAccessControlHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the file or directory. */
    etag?: string;
    /** The data and time the file or directory was last modified. Write operations on the file or directory update the last modified time. */
    lastModified?: Date;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
}

/** Optional parameters. */
declare interface PathSetAccessControlOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions_2;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Optional and only valid if Hierarchical Namespace is enabled for the account. Sets POSIX access permissions for the file owner, the file owning group, and others. Each class may be granted read, write, or execute permission.  The sticky bit is also supported.  Both symbolic (rwxrw-rw-) and 4-digit octal notation (e.g. 0766) are supported. */
    permissions?: string;
    /** Optional. The owner of the blob or directory. */
    owner?: string;
    /** Optional. The owning group of the blob or directory. */
    group?: string;
    /** Sets POSIX access control rights on files and directories. The value is a comma-separated list of access control entries. Each access control entry (ACE) consists of a scope, a type, a user or group identifier, and permissions in the format "[scope:][type]:[id]:[permissions]". */
    acl?: string;
}

export declare interface PathSetAccessControlOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
    owner?: string;
    group?: string;
}

/** Defines headers for Path_setAccessControlRecursive operation. */
declare interface PathSetAccessControlRecursiveHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** When performing setAccessControlRecursive on a directory, the number of paths that are processed with each invocation is limited.  If the number of paths to be processed exceeds this limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the setAccessControlRecursive operation to continue the setAccessControlRecursive operation on the directory. */
    continuation?: string;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
}

/** Defines values for PathSetAccessControlRecursiveMode. */
declare type PathSetAccessControlRecursiveMode = "set" | "modify" | "remove";

/** Optional parameters. */
declare interface PathSetAccessControlRecursiveOptionalParams extends coreClient.OperationOptions {
    /** Optional.  When deleting a directory, the number of paths that are deleted with each invocation is limited.  If the number of paths to be deleted exceeds this limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the delete operation to continue deleting the directory. */
    continuation?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Sets POSIX access control rights on files and directories. The value is a comma-separated list of access control entries. Each access control entry (ACE) consists of a scope, a type, a user or group identifier, and permissions in the format "[scope:][type]:[id]:[permissions]". */
    acl?: string;
    /** Optional. It specifies the maximum number of files or directories on which the acl change will be applied. If omitted or greater than 2,000, the request will process up to 2,000 items */
    maxRecords?: number;
    /** Optional. Valid for "SetAccessControlRecursive" operation. If set to false, the operation will terminate quickly on encountering user errors (4XX). If true, the operation will ignore user errors and proceed with the operation on other sub-entities of the directory. Continuation token will only be returned when forceFlag is true in case of user errors. If not set the default value is false for this. */
    forceFlag?: boolean;
}

/** Contains response data for the setAccessControlRecursive operation. */
declare type PathSetAccessControlRecursiveResponse = PathSetAccessControlRecursiveHeaders & SetAccessControlRecursiveResponse;

export declare type PathSetAccessControlResponse = WithResponse<PathSetAccessControlHeaders, PathSetAccessControlHeaders>;

/** Contains response data for the setAccessControl operation. */
declare type PathSetAccessControlResponse_2 = PathSetAccessControlHeaders;

/** Optional parameters. */
declare interface PathSetExpiryOptionalParams extends coreClient.OperationOptions {
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** The time to set the blob to expiry */
    expiresOn?: string;
}

/** Contains response data for the setExpiry operation. */
declare type PathSetExpiryResponse = FileSetExpiryHeaders;

export declare interface PathSetHttpHeadersHeaders {
    etag?: string;
    lastModified?: Date;
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
}

export declare interface PathSetHttpHeadersOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
}

export declare type PathSetHttpHeadersResponse = WithResponse<PathSetHttpHeadersHeaders, PathSetHttpHeadersHeaders>;

export declare interface PathSetMetadataHeaders {
    etag?: string;
    lastModified?: Date;
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
    isServerEncrypted?: boolean;
    encryptionKeySha256?: string;
}

export declare interface PathSetMetadataOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
    /**
     * Customer Provided Key Info.
     */
    customerProvidedKey?: CpkInfo;
}

export declare type PathSetMetadataResponse = WithResponse<PathSetMetadataHeaders, PathSetMetadataHeaders>;

export declare interface PathSetPermissionsOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    conditions?: DataLakeRequestConditions;
    owner?: string;
    group?: string;
}

export declare type PathSetPermissionsResponse = WithResponse<PathSetAccessControlHeaders, PathSetAccessControlHeaders>;

/** Defines headers for Path_undelete operation. */
export declare interface PathUndeleteHeaders {
    /** If a client request id header is sent in the request, this header will be present in the response with the same value. */
    clientRequestId?: string;
    /** This header uniquely identifies the request that was made and can be used for troubleshooting the request. */
    requestId?: string;
    /** The type of the resource.  The value may be "file" or "directory".  If not set, the value is "file". */
    resourceType?: string;
    /** Indicates the version of the Blob service used to execute the request. This header is returned for requests made against version 2009-09-19 and above. */
    version?: string;
    /** UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
}

/** Optional parameters. */
declare interface PathUndeleteOptionalParams extends coreClient.OperationOptions {
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Only for hierarchical namespace enabled accounts. Optional. The path of the soft deleted blob to undelete. */
    undeleteSource?: string;
}

/** Contains response data for the undelete operation. */
declare type PathUndeleteResponse = PathUndeleteHeaders;

/** Defines values for PathUpdateAction. */
declare type PathUpdateAction = "append" | "flush" | "setProperties" | "setAccessControl" | "setAccessControlRecursive";

/** Defines headers for Path_update operation. */
export declare interface PathUpdateHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** An HTTP entity tag associated with the file or directory. */
    etag?: string;
    /** The data and time the file or directory was last modified.  Write operations on the file or directory update the last modified time. */
    lastModified?: Date;
    /** Indicates that the service supports requests for partial file content. */
    acceptRanges?: string;
    /** If the Cache-Control request header has previously been set for the resource, that value is returned in this header. */
    cacheControl?: string;
    /** If the Content-Disposition request header has previously been set for the resource, that value is returned in this header. */
    contentDisposition?: string;
    /** If the Content-Encoding request header has previously been set for the resource, that value is returned in this header. */
    contentEncoding?: string;
    /** If the Content-Language request header has previously been set for the resource, that value is returned in this header. */
    contentLanguage?: string;
    /** The size of the resource in bytes. */
    contentLength?: number;
    /** Indicates the range of bytes returned in the event that the client requested a subset of the file by setting the Range request header. */
    contentRange?: string;
    /** The content type specified for the resource. If no content type was specified, the default content type is application/octet-stream. */
    contentType?: string;
    /** An MD5 hash of the request content. This header is only returned for "Flush" operation. This header is returned so that the client can check for message content integrity. This header refers to the content of the request, not actual file content. */
    contentMD5?: string;
    /** User-defined properties associated with the file or directory, in the format of a comma-separated list of name and value pairs "n1=v1, n2=v2, ...", where each value is a base64 encoded string. Note that the string may only contain ASCII characters in the ISO-8859-1 character set. */
    properties?: string;
    /** When performing setAccessControlRecursive on a directory, the number of paths that are processed with each invocation is limited.  If the number of paths to be processed exceeds this limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the setAccessControlRecursive operation to continue the setAccessControlRecursive operation on the directory. */
    xMsContinuation?: string;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** Error Code */
    errorCode?: string;
}

/** Optional parameters. */
declare interface PathUpdateOptionalParams extends coreClient.OperationOptions {
    /** Parameter group */
    modifiedAccessConditions?: ModifiedAccessConditions_2;
    /** Parameter group */
    pathHttpHeaders?: PathHttpHeaders_2;
    /** Parameter group */
    leaseAccessConditions?: LeaseAccessConditions_2;
    /** Optional. The number of paths processed with each invocation is limited. If the number of paths to be processed exceeds this limit, a continuation token is returned in the response header x-ms-continuation. When a continuation token is  returned in the response, it must be percent-encoded and specified in a subsequent invocation of setAccessControlRecursive operation. */
    continuation?: string;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
    /** Optional. User-defined properties to be stored with the filesystem, in the format of a comma-separated list of name and value pairs "n1=v1, n2=v2, ...", where each value is a base64 encoded string. Note that the string may only contain ASCII characters in the ISO-8859-1 character set.  If the filesystem exists, any properties not included in the list will be removed.  All properties are removed if the header is omitted.  To merge new and existing properties, first get all existing properties and the current E-Tag, then make a conditional request with the E-Tag and include values for all properties. */
    properties?: string;
    /** Optional and only valid if Hierarchical Namespace is enabled for the account. Sets POSIX access permissions for the file owner, the file owning group, and others. Each class may be granted read, write, or execute permission.  The sticky bit is also supported.  Both symbolic (rwxrw-rw-) and 4-digit octal notation (e.g. 0766) are supported. */
    permissions?: string;
    /** Optional. The owner of the blob or directory. */
    owner?: string;
    /** Optional. The owning group of the blob or directory. */
    group?: string;
    /** Sets POSIX access control rights on files and directories. The value is a comma-separated list of access control entries. Each access control entry (ACE) consists of a scope, a type, a user or group identifier, and permissions in the format "[scope:][type]:[id]:[permissions]". */
    acl?: string;
    /** Optional. Valid for "SetAccessControlRecursive" operation. It specifies the maximum number of files or directories on which the acl change will be applied. If omitted or greater than 2,000, the request will process up to 2,000 items */
    maxRecords?: number;
    /** Optional. Valid for "SetAccessControlRecursive" operation. If set to false, the operation will terminate quickly on encountering user errors (4XX). If true, the operation will ignore user errors and proceed with the operation on other sub-entities of the directory. Continuation token will only be returned when forceFlag is true in case of user errors. If not set the default value is false for this. */
    forceFlag?: boolean;
    /** This parameter allows the caller to upload data in parallel and control the order in which it is appended to the file.  It is required when uploading data to be appended to the file and when flushing previously uploaded data to the file.  The value must be the position where the data is to be appended.  Uploaded data is not immediately flushed, or written, to the file.  To flush, the previously uploaded data must be contiguous, the position parameter must be specified and equal to the length of the file after all data has been written, and there must not be a request entity body included with the request. */
    position?: number;
    /** Valid only for flush operations.  If "true", uncommitted data is retained after the flush operation completes; otherwise, the uncommitted data is deleted after the flush operation.  The default is false.  Data at offsets less than the specified position are written to the file when flush succeeds, but this optional parameter allows data after the flush position to be retained for a future flush operation. */
    retainUncommittedData?: boolean;
    /** Azure Storage Events allow applications to receive notifications when files change. When Azure Storage Events are enabled, a file changed event is raised. This event has a property indicating whether this is the final change to distinguish the difference between an intermediate flush to a file stream and the final close of a file stream. The close query parameter is valid only when the action is "flush" and change notifications are enabled. If the value of close is "true" and the flush operation completes successfully, the service raises a file change notification with a property indicating that this is the final update (the file stream has been closed). If "false" a change notification is raised indicating the file has changed. The default is false. This query parameter is set to true by the Hadoop ABFS driver to indicate that the file stream has been closed." */
    close?: boolean;
    /** Required for "Append Data" and "Flush Data".  Must be 0 for "Flush Data".  Must be the length of the request content in bytes for "Append Data". */
    contentLength?: number;
}

/** Contains response data for the update operation. */
declare type PathUpdateResponse = PathUpdateHeaders & SetAccessControlRecursiveResponse;

export { Pipeline }

export { PipelineLike }

export { PipelineOptions }

export declare type PublicAccessType = "filesystem" | "file";

export declare interface RawAccessPolicy {
    startsOn?: string;
    expiresOn?: string;
    permissions: string;
}

export declare interface RemovePathAccessControlItem {
    /**
     * Indicates whether this is the default entry for the ACL.
     */
    defaultScope: boolean;
    /**
     * Specifies which role this entry targets.
     */
    accessControlType: AccessControlType;
    /**
     * Specifies the entity for which this entry applies.
     * Must be omitted for types mask or other. It must also be omitted when the user or group is the owner.
     */
    entityId?: string;
}

export { RequestPolicy as IHttpClient }
export { RequestPolicy }

export { RequestPolicyFactory }

export { RequestPolicyOptions }

export { RestError }

export declare interface RolePermissions {
    read: boolean;
    write: boolean;
    execute: boolean;
}

/**
 * Allowed IP range for a SAS.
 */
export declare interface SasIPRange {
    /**
     * Starting IP address in the IP range.
     * If end IP doesn't provide, start IP will the only IP allowed.
     */
    start: string;
    /**
     * Optional. IP address that ends the IP range.
     * If not provided, start IP will the only IP allowed.
     */
    end?: string;
}

/**
 * Protocols for generated SAS.
 */
export declare enum SASProtocol {
    /**
     * Protocol that allows HTTPS only
     */
    Https = "https",
    /**
     * Protocol that allows both HTTPS and HTTP
     */
    HttpsAndHttp = "https,http"
}

/**
 * Represents the components that make up an Azure Storage SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link AccountSASSignatureValues} and {@link BlobSASSignatureValues}
 * types. Once generated, it can be encoded into a {@link String} and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 */
export declare class SASQueryParameters {
    /**
     * The storage API version.
     */
    readonly version: string;
    /**
     * Optional. The allowed HTTP protocol(s).
     */
    readonly protocol?: SASProtocol;
    /**
     * Optional. The start time for this SAS token.
     */
    readonly startsOn?: Date;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     */
    readonly expiresOn?: Date;
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSASPermissions}, {@link BlobSASPermissions}, or {@link ContainerSASPermissions} for
     * more details.
     */
    readonly permissions?: string;
    /**
     * Optional. The storage services being accessed (only for Account SAS). Please refer to {@link AccountSASServices}
     * for more details.
     */
    readonly services?: string;
    /**
     * Optional. The storage resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSASResourceTypes} for more details.
     */
    readonly resourceTypes?: string;
    /**
     * Optional. The signed identifier (only for {@link BlobSASSignatureValues}).
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/establishing-a-stored-access-policy
     */
    readonly identifier?: string;
    /**
     * Optional. Specifies which resources are accessible via the SAS (only for {@link BlobSASSignatureValues}).
     * @see https://learn.microsoft.com/rest/api/storageservices/create-service-sas#specifying-the-signed-resource-blob-service-only
     */
    readonly resource?: string;
    /**
     * The signature for the SAS token.
     */
    readonly signature: string;
    /**
     * Value for cache-control header in Blob/File Service SAS.
     */
    readonly cacheControl?: string;
    /**
     * Value for content-disposition header in Blob/File Service SAS.
     */
    readonly contentDisposition?: string;
    /**
     * Value for content-encoding header in Blob/File Service SAS.
     */
    readonly contentEncoding?: string;
    /**
     * Value for content-length header in Blob/File Service SAS.
     */
    readonly contentLanguage?: string;
    /**
     * Value for content-type header in Blob/File Service SAS.
     */
    readonly contentType?: string;
    /**
     * Inner value of getter ipRange.
     */
    private readonly ipRangeInner?;
    /**
     * The Azure Active Directory object ID in GUID format.
     * Property of user delegation key.
     */
    private readonly signedOid?;
    /**
     * The Azure Active Directory tenant ID in GUID format.
     * Property of user delegation key.
     */
    private readonly signedTenantId?;
    /**
     * The date-time the key is active.
     * Property of user delegation key.
     */
    private readonly signedStartsOn?;
    /**
     * The date-time the key expires.
     * Property of user delegation key.
     */
    private readonly signedExpiresOn?;
    /**
     * Abbreviation of the Azure Storage service that accepts the user delegation key.
     * Property of user delegation key.
     */
    private readonly signedService?;
    /**
     * The service version that created the user delegation key.
     * Property of user delegation key.
     */
    private readonly signedVersion?;
    /**
     * Indicate the depth of the directory specified in the canonicalizedresource field of the string-to-sign.
     * The depth of the directory is the number of directories beneath the root folder.
     */
    readonly directoryDepth?: number;
    /**
     * Authorized AAD Object ID in GUID format. The AAD Object ID of a user authorized by the owner of the User Delegation Key
     * to perform the action granted by the SAS. The Azure Storage service will ensure that the owner of the user delegation key
     * has the required permissions before granting access but no additional permission check for the user specified in
     * this value will be performed. This cannot be used in conjuction with {@link signedUnauthorizedUserObjectId}.
     * This is only used for User Delegation SAS.
     */
    readonly preauthorizedAgentObjectId?: string;
    /**
     * Unauthorized AAD Object ID in GUID format. The AAD Object ID of a user that is assumed to be unauthorized by the owner of the User Delegation Key.
     * The Azure Storage Service will perform an additional POSIX ACL check to determine if the user is authorized to perform the requested operation.
     * This cannot be used in conjuction with {@link signedAuthorizedUserObjectId}.
     * This is only used for User Delegation SAS.
     */
    readonly agentObjectId?: string;
    /**
     * A GUID value that will be logged in the storage diagnostic logs and can be used to correlate SAS generation with storage resource access.
     * This is only used for User Delegation SAS.
     */
    readonly correlationId?: string;
    /**
     * Optional. Encryption scope to use when sending requests authorized with this SAS URI.
     */
    readonly encryptionScope?: string;
    /**
     * Optional. IP range allowed for this SAS.
     *
     * @readonly
     */
    get ipRange(): SasIPRange | undefined;
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param version - Representing the storage version
     * @param signature - Representing the signature for the SAS token
     * @param permissions - Representing the storage permissions
     * @param services - Representing the storage services being accessed (only for Account SAS)
     * @param resourceTypes - Representing the storage resource types being accessed (only for Account SAS)
     * @param protocol - Representing the allowed HTTP protocol(s)
     * @param startsOn - Representing the start time for this SAS token
     * @param expiresOn - Representing the expiry time for this SAS token
     * @param ipRange - Representing the range of valid IP addresses for this SAS token
     * @param identifier - Representing the signed identifier (only for Service SAS)
     * @param resource - Representing the storage container or blob (only for Service SAS)
     * @param cacheControl - Representing the cache-control header (only for Blob/File Service SAS)
     * @param contentDisposition - Representing the content-disposition header (only for Blob/File Service SAS)
     * @param contentEncoding - Representing the content-encoding header (only for Blob/File Service SAS)
     * @param contentLanguage - Representing the content-language header (only for Blob/File Service SAS)
     * @param contentType - Representing the content-type header (only for Blob/File Service SAS)
     * @param userDelegationKey - Representing the user delegation key properties
     * @param preauthorizedAgentObjectId - Representing the authorized AAD Object ID (only for User Delegation SAS)
     * @param agentObjectId - Representing the unauthorized AAD Object ID (only for User Delegation SAS)
     * @param correlationId - Representing the correlation ID (only for User Delegation SAS)
     */
    constructor(version: string, signature: string, permissions?: string, services?: string, resourceTypes?: string, protocol?: SASProtocol, startsOn?: Date, expiresOn?: Date, ipRange?: SasIPRange, identifier?: string, resource?: string, cacheControl?: string, contentDisposition?: string, contentEncoding?: string, contentLanguage?: string, contentType?: string, userDelegationKey?: UserDelegationKey, directoryDepth?: number, preauthorizedAgentObjectId?: string, agentObjectId?: string, correlationId?: string, encryptionScope?: string);
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param version - Representing the storage version
     * @param signature - Representing the signature for the SAS token
     * @param options - Optional. Options to construct the SASQueryParameters.
     */
    constructor(version: string, signature: string, options?: SASQueryParametersOptions);
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     */
    toString(): string;
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @param queries -
     * @param key -
     * @param value -
     */
    private tryAppendQueryParameter;
}

/**
 * Options to construct {@link SASQueryParameters}.
 */
export declare interface SASQueryParametersOptions {
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSASPermissions}, {@link BlobSASPermissions}, or {@link ContainerSASPermissions} for
     * more details.
     */
    permissions?: string;
    /**
     * Optional. The storage services being accessed (only for Account SAS). Please refer to {@link AccountSASServices}
     * for more details.
     */
    services?: string;
    /**
     * Optional. The storage resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSASResourceTypes} for more details.
     */
    resourceTypes?: string;
    /**
     * Optional. The allowed HTTP protocol(s).
     */
    protocol?: SASProtocol;
    /**
     * Optional. The start time for this SAS token.
     */
    startsOn?: Date;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     */
    expiresOn?: Date;
    /**
     * Optional. IP ranges allowed in this SAS.
     */
    ipRange?: SasIPRange;
    /**
     * Optional. The signed identifier (only for {@link BlobSASSignatureValues}).
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/establishing-a-stored-access-policy
     */
    identifier?: string;
    /**
     * Optional. Encryption scope to use when sending requests authorized with this SAS URI.
     */
    encryptionScope?: string;
    /**
     * Optional. Specifies which resources are accessible via the SAS (only for {@link BlobSASSignatureValues}).
     * @see https://learn.microsoft.com/rest/api/storageservices/create-service-sas#specifying-the-signed-resource-blob-service-only
     */
    resource?: string;
    /**
     * Value for cache-control header in Blob/File Service SAS.
     */
    cacheControl?: string;
    /**
     * Value for content-disposition header in Blob/File Service SAS.
     */
    contentDisposition?: string;
    /**
     * Value for content-encoding header in Blob/File Service SAS.
     */
    contentEncoding?: string;
    /**
     * Value for content-length header in Blob/File Service SAS.
     */
    contentLanguage?: string;
    /**
     * Value for content-type header in Blob/File Service SAS.
     */
    contentType?: string;
    /**
     * User delegation key properties.
     */
    userDelegationKey?: UserDelegationKey;
    /**
     * Indicate the depth of the directory specified in the canonicalizedresource field of the string-to-sign.
     * The depth of the directory is the number of directories beneath the root folder.
     */
    directoryDepth?: number;
    /**
     * Authorized AAD Object ID in GUID format. The AAD Object ID of a user authorized by the owner of the User Delegation Key
     * to perform the action granted by the SAS. The Azure Storage service will ensure that the owner of the user delegation key
     * has the required permissions before granting access but no additional permission check for the user specified in
     * this value will be performed. This cannot be used in conjuction with {@link signedUnauthorizedUserObjectId}.
     * This is only used for User Delegation SAS.
     */
    preauthorizedAgentObjectId?: string;
    /**
     * Unauthorized AAD Object ID in GUID format. The AAD Object ID of a user that is assumed to be unauthorized by the owner of the User Delegation Key.
     * The Azure Storage Service will perform an additional POSIX ACL check to determine if the user is authorized to perform the requested operation.
     * This cannot be used in conjuction with {@link signedAuthorizedUserObjectId}. This is only used for User Delegation SAS.
     */
    agentObjectId?: string;
    /**
     * A GUID value that will be logged in the storage diagnostic logs and can be used to correlate SAS generation with storage resource access.
     * This is only used for User Delegation SAS.
     */
    correlationId?: string;
}

/** Interface representing a Service. */
declare interface Service {
    /**
     * List filesystems and their properties in given account.
     * @param options The options parameters.
     */
    listFileSystems(options?: ServiceListFileSystemsOptionalParams): Promise<ServiceListFileSystemsResponse>;
}

export { ServiceClientOptions }

/**
 * Options to configure {@link DataLakeServiceClient.generateAccountSasUrl} operation.
 */
export declare interface ServiceGenerateAccountSasUrlOptions {
    /**
     * The version of the service this SAS will target. If not specified, it will default to the version targeted by the
     * library.
     */
    version?: string;
    /**
     * Optional. SAS protocols allowed.
     */
    protocol?: SASProtocol;
    /**
     * Optional. When the SAS will take effect.
     */
    startsOn?: Date;
    /**
     * Optional. IP range allowed.
     */
    ipRange?: SasIPRange;
    /**
     * Optional. Encryption scope to use when sending requests authorized with this SAS URI.
     */
    encryptionScope?: string;
}

export declare interface ServiceGetUserDelegationKeyHeaders {
    clientRequestId?: string;
    requestId?: string;
    version?: string;
    date?: Date;
}

/** ***********************************************************/
/** DataLakeServiceClient option and response related models */
/** ***********************************************************/
export declare interface ServiceGetUserDelegationKeyOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
}

export declare type ServiceGetUserDelegationKeyResponse = WithResponse<UserDelegationKey & ServiceGetUserDelegationKeyHeaders, ServiceGetUserDelegationKeyHeaders, UserDelegationKeyModel>;

export { ServiceListContainersSegmentResponse }

/** Defines headers for Service_listFileSystems operation. */
declare interface ServiceListFileSystemsHeaders {
    /** A UTC date/time value generated by the service that indicates the time at which the response was initiated. */
    date?: Date;
    /** A server-generated UUID recorded in the analytics logs for troubleshooting and correlation. */
    requestId?: string;
    /** The version of the REST protocol used to process the request. */
    version?: string;
    /** If the number of filesystems to be listed exceeds the maxResults limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the list operation to continue listing the filesystems. */
    continuation?: string;
    /** The content type of list filesystem response. The default content type is application/json. */
    contentType?: string;
}

/** Optional parameters. */
declare interface ServiceListFileSystemsOptionalParams extends coreClient.OperationOptions {
    /** Filters results to filesystems within the specified prefix. */
    prefix?: string;
    /** Optional.  When deleting a directory, the number of paths that are deleted with each invocation is limited.  If the number of paths to be deleted exceeds this limit, a continuation token is returned in this response header.  When a continuation token is returned in the response, it must be specified in a subsequent invocation of the delete operation to continue deleting the directory. */
    continuation?: string;
    /** An optional value that specifies the maximum number of items to return. If omitted or greater than 5,000, the response will include up to 5,000 items. */
    maxResults?: number;
    /** Provides a client-generated, opaque value with a 1 KB character limit that is recorded in the analytics logs when storage analytics logging is enabled. */
    requestId?: string;
    /** The timeout parameter is expressed in seconds. For more information, see <a href="https://docs.microsoft.com/en-us/rest/api/storageservices/fileservices/setting-timeouts-for-blob-service-operations">Setting Timeouts for Blob Service Operations.</a> */
    timeout?: number;
}

export declare interface ServiceListFileSystemsOptions extends CommonOptions {
    abortSignal?: AbortSignalLike;
    prefix?: string;
    includeMetadata?: boolean;
    /**
     * Specifies whether soft deleted File System should be included in the response.
     */
    includeDeleted?: boolean;
}

/** Contains response data for the listFileSystems operation. */
declare type ServiceListFileSystemsResponse = ServiceListFileSystemsHeaders & FileSystemList;

export declare interface ServiceListFileSystemsSegmentHeaders {
    clientRequestId?: string;
    requestId?: string;
    version?: string;
}

export declare type ServiceListFileSystemsSegmentResponse = WithResponse<ListFileSystemsSegmentResponse & ServiceListFileSystemsSegmentHeaders, ServiceListFileSystemsSegmentHeaders, ListFileSystemsSegmentResponse>;

/**
 * Options to configure {@link DataLakeServiceClient.renameFileSystem}.
 */
export declare type ServiceRenameFileSystemOptions = ServiceRenameContainerOptions;

/**
 * Options to configure {@link DataLakeServiceClient.undeleteFileSystem}.
 */
export declare interface ServiceUndeleteFileSystemOptions extends CommonOptions {
    /**
     * An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Optional. Specifies the new name of the restored File System.
     * Will use its original name if this is not specified.
     * @deprecated Restore FileSystem to a different name is not supported by service anymore.
     */
    destinationFileSystemName?: string;
}

declare interface SetAccessControlRecursiveResponse {
    directoriesSuccessful?: number;
    filesSuccessful?: number;
    failureCount?: number;
    failedEntries?: AclFailedEntry[];
}

export declare interface SignedIdentifier<T> {
    id: string;
    accessPolicy: T;
}

/** Parameter group */
declare interface SourceModifiedAccessConditions {
    /** Specify an ETag value to operate only on blobs with a matching value. */
    sourceIfMatch?: string;
    /** Specify an ETag value to operate only on blobs without a matching value. */
    sourceIfNoneMatch?: string;
    /** Specify this header value to operate only on a blob if it has been modified since the specified date/time. */
    sourceIfModifiedSince?: Date;
    /** Specify this header value to operate only on a blob if it has not been modified since the specified date/time. */
    sourceIfUnmodifiedSince?: Date;
}

export { StorageBrowserPolicy }

export { StorageBrowserPolicyFactory }

/**
 * A StorageClient represents a based URL class for {@link BlobServiceClient}, {@link ContainerClient}
 * and etc.
 */
declare abstract class StorageClient {
    /**
     * Encoded URL string value.
     */
    readonly url: string;
    readonly accountName: string;
    /**
     * Encoded URL string value for corresponding blob endpoint.
     */
    protected readonly blobEndpointUrl: string;
    /**
     * Encoded URL string value for corresponding dfs endpoint.
     */
    protected readonly dfsEndpointUrl: string;
    /* Excluded from this release type: pipeline */
    /**
     * Such as AnonymousCredential, StorageSharedKeyCredential or any credential from the `@azure/identity` package to authenticate requests to the service. You can also provide an object that implements the TokenCredential interface. If not specified, AnonymousCredential is used.
     */
    readonly credential: StorageSharedKeyCredential | AnonymousCredential | TokenCredential;
    /**
     * StorageClient is a reference to protocol layer operations entry, which is
     * generated by AutoRest generator.
     */
    protected readonly storageClientContext: StorageClient_2;
    /**
     * storageClientContextWithBlobEndpoint is a reference to protocol layer operations entry, which is
     * generated by AutoRest generator, with its url pointing to the Blob endpoint.
     */
    protected readonly storageClientContextToBlobEndpoint: StorageClient_2;
    /**
     */
    protected readonly isHttps: boolean;
    /**
     * Creates an instance of StorageClient.
     * @param url - url to resource
     * @param pipeline - request policy pipeline.
     */
    protected constructor(url: string, pipeline: PipelineLike);
}

declare class StorageClient_2 extends coreHttpCompat.ExtendedServiceClient {
    url: string;
    version: string;
    resource: string;
    xMsLeaseDuration?: number;
    /**
     * Initializes a new instance of the StorageClient class.
     * @param url The URL of the service account, container, or blob that is the target of the desired
     *            operation.
     * @param options The parameter options
     */
    constructor(url: string, options?: StorageClientOptionalParams);
    service: Service;
    fileSystemOperations: FileSystemOperations;
    pathOperations: PathOperations;
}

/** Optional parameters. */
declare interface StorageClientOptionalParams extends coreHttpCompat.ExtendedServiceClientOptions {
    /** Specifies the version of the operation to use for this request. */
    version?: string;
    /** The value must be "filesystem" for all filesystem operations. */
    resource?: string;
    /** The lease duration is required to acquire a lease, and specifies the duration of the lease in seconds.  The lease duration must be between 15 and 60 seconds or -1 for infinite lease. */
    xMsLeaseDuration?: number;
    /** Overrides client endpoint. */
    endpoint?: string;
}

/**
 * Defines the known cloud audiences for Storage.
 */
export declare enum StorageDataLakeAudience {
    /**
     * The OAuth scope to use to retrieve an AAD token for Azure Storage.
     */
    StorageOAuthScopes = "https://storage.azure.com/.default"
}

/**
 * The OAuth scope to use with Azure Storage.
 */
export declare const StorageOAuthScopes: string | string[];

/**
 * Options interface for the {@link newPipeline} function.
 */
export declare interface StoragePipelineOptions {
    /**
     * Options to configure a proxy for outgoing requests.
     */
    proxyOptions?: ProxySettings;
    /**
     * Options for adding user agent details to outgoing requests.
     */
    userAgentOptions?: UserAgentPolicyOptions;
    /**
     * Configures the built-in retry policy behavior.
     */
    retryOptions?: StorageRetryOptions;
    /**
     * Keep alive configurations. Default keep-alive is enabled.
     */
    keepAliveOptions?: KeepAliveOptions;
    /**
     * Configures the HTTP client to send requests and receive responses.
     */
    httpClient?: RequestPolicy;
    /**
     * The audience used to retrieve an AAD token.
     * By default, audience 'https://storage.azure.com/.default' will be used.
     */
    audience?: string;
}

export { StorageRetryOptions }

export { StorageRetryPolicy }

export { StorageRetryPolicyFactory }

export { StorageRetryPolicyType }

export { StorageSharedKeyCredential }

export { StorageSharedKeyCredentialPolicy }

export declare const ToBlobEndpointHostMappings: string[][];

export declare const ToDfsEndpointHostMappings: string[][];

export declare interface UserDelegationKey {
    signedObjectId: string;
    signedTenantId: string;
    signedStartsOn: Date;
    signedExpiresOn: Date;
    signedService: string;
    signedVersion: string;
    value: string;
}

export { UserDelegationKeyModel }

export { WebResource }

export { }
